unit AddInObj;

interface

uses
  Windows, Classes, ComServ, ComObj, ActiveX, SysUtils, Variants,
  AddInLib;

const
CLSID_AddInObject : TGUID = '{B3A1AA50-D7AB-4F3A-8BF4-2AA0E3DF74A4}';

resourcestring
// 
strFileName     = 'FileName';
strEOF          = 'EOF';
strSize         = 'Size';
strPos          = 'Pos';
strDebugMode    = 'DebugMode';
strErrorCode    = 'ErrorCode';
strErrorDescr   = 'ErrorDescription';

// 
strCreate       = 'Create';
strOpen         = 'Open';
strClose        = 'Close';
strRead         = 'Read';
strWrite        = 'Write';
strReadBin      = 'ReadBin';
strWriteBin     = 'WriteBin';
strReadSingle   = 'ReadSingle';
strWriteSingle  = 'WriteSingle';
strReadDouble   = 'ReadDouble';
strWriteDouble  = 'WriteDouble';
strReadCurrency = 'ReadCurrency';
strWriteCurrency= 'WriteCurrency';
strReadHex      = 'ReadHex';
strWriteHex     = 'WriteHex';
strStrToHex     = 'StrToHex';
strHexToStr     = 'HexToStr';
strCompress     = 'Compress';
strDecompress   = 'Decompress';

const
MaxParamCount = 5;

type
TPropertyRec = record
  Name: String;
  Readable: Boolean;
  Writable: Boolean;
end;

TParamRec = record
  Name: String;
  case HasDefValue: (dvNo, dvNull, dvInt, dvReal, dvStr) of
    dvInt:  (DefInt: Integer);
    dvReal: (DefReal: Double);
    dvStr:  (DefStr: String[20]);
end;

TMethodRec = record
  Name: String;
  HasRetValue: Boolean;
  ParamCount: 0..MaxParamCount;
  Params: array [0..MaxParamCount-1] of TParamRec;
end;

const
Props: array [0..6] of TPropertyRec = (
  (Name: strFileName;   Readable: True;   Writable: False),
  (Name: strEOF;        Readable: True;   Writable: False),
  (Name: strSize;       Readable: True;   Writable: False),
  (Name: strPos;        Readable: True;   Writable: True),
  (Name: strDebugMode;  Readable: True;   Writable: True),
  (Name: strErrorCode;  Readable: True;   Writable: False),
  (Name: strErrorDescr; Readable: True;   Writable: False));

Methods: array [0..18] of TMethodRec = (
  (Name: strCreate;     HasRetValue: False; ParamCount: 1; Params: ((Name: 'FName';  HasDefValue: dvNo),(),(),(),())),
  (Name: strOpen;       HasRetValue: False; ParamCount: 2; Params: ((Name: 'FName';  HasDefValue: dvNo),
                                                                    (Name: 'Mode';   HasDefValue: dvInt; DefInt: 2),(),(),())),
  (Name: strClose;      HasRetValue: False; ParamCount: 0),
//----------------------------------------------------------------------------
  (Name: strRead;       HasRetValue: True;  ParamCount: 2; Params: ((Name: 'Count';  HasDefValue: dvInt; DefInt: -1),
                                                                    (Name: 'ZeroStr';HasDefValue: dvStr; DefStr: ' '),(),(),())),
  (Name: strWrite;      HasRetValue: False; ParamCount: 3; Params: ((Name: 'Str';    HasDefValue: dvNo),
                                                                    (Name: 'Count';  HasDefValue: dvInt; DefInt: -1),
                                                                    (Name: 'FillCh'; HasDefValue: dvInt; DefInt: 32),(),())),
  (Name: strReadBin;    HasRetValue: True;  ParamCount: 1; Params: ((Name: 'Count';  HasDefValue: dvInt; DefInt: 4),(),(),(),())),
  (Name: strWriteBin;   HasRetValue: False; ParamCount: 2; Params: ((Name: 'Number'; HasDefValue: dvNo),
                                                                    (Name: 'Count';  HasDefValue: dvInt; DefInt: 4),(),(),())),
  (Name: strReadSingle; HasRetValue: True;  ParamCount: 0),
  (Name: strWriteSingle;HasRetValue: False; ParamCount: 1; Params: ((Name: 'Value';  HasDefValue: dvNo),(),(),(),())),
  (Name: strReadDouble; HasRetValue: True;  ParamCount: 0),
  (Name: strWriteDouble; HasRetValue: False; ParamCount: 1; Params: ((Name: 'Value';  HasDefValue: dvNo),(),(),(),())),
  (Name: strReadCurrency; HasRetValue: True;  ParamCount: 0),
  (Name: strWriteCurrency; HasRetValue: False; ParamCount: 1; Params: ((Name: 'Value';  HasDefValue: dvNo),(),(),(),())),
  (Name: strReadHex;    HasRetValue: True;  ParamCount: 1; Params: ((Name: 'Count';  HasDefValue: dvInt; DefInt: -1),(),(),(),())),
  (Name: strWriteHex;   HasRetValue: False; ParamCount: 1; Params: ((Name: 'Str';    HasDefValue: dvNo),(),(),(),())),
//----------------------------------------------------------------------------
  (Name: strStrToHex;   HasRetValue: True;  ParamCount: 1; Params: ((Name: 'Str';    HasDefValue: dvNo),(),(),(),())),
  (Name: strHexToStr;   HasRetValue: True;  ParamCount: 1; Params: ((Name: 'Str';    HasDefValue: dvNo),(),(),(),())),
//----------------------------------------------------------------------------
  (Name: strCompress;   HasRetValue: True;  ParamCount: 1; Params: ((Name: 'Str';    HasDefValue: dvNo),(),(),(),())),
  (Name: strDecompress; HasRetValue: True;  ParamCount: 1; Params: ((Name: 'Str';    HasDefValue: dvNo),(),(),(),())));

type
TAddInObject = class (TComObject, IInitDone, ILanguageExtender)
      { Attributes }
  public
    FileName: string;
    DebugMode: Boolean;
    ErrorCode: Integer;

      { Interfaces }
    pErrorLog : IErrorLog;
    pEvent : IAsyncEvent;
    pProfile : IPropertyProfile;
    pStatusLine : IStatusLine;

  protected
    FHandle: File;
    FMode: Integer;  //-1 - closed, 0 - read, 1 - write, 2 - rw

    function LoadProperties: Boolean;
    procedure SaveProperties;
      {These two methods is convenient way to access function
       parameters from SAFEARRAY vector of variants }
    function GetNParam(var pArray : PSafeArray; lIndex: Integer ): OleVariant;
    procedure PutNParam(var pArray: PSafeArray; lIndex: Integer; var varPut: OleVariant);
      {1C Message log}
    procedure AddMsg(const Description: string; const wIcon: Word = ADDIN_E_ORDINARY);
      { Interface implementation }
      { IInitDone implementation }
    function Init(const pConnection: IDispatch): HResult; stdcall;
    function Done: HResult; stdcall;
    function GetInfo(var pInfo: PSafeArray{(OleVariant)}): HResult; stdcall;
      { ILanguageExtender implementation }
    function RegisterExtensionAs(var bstrExtensionName: WideString): HResult; stdcall;
    function GetNProps(var plProps: Integer): HResult; stdcall;
    function FindProp(const bstrPropName: WideString; var plPropNum: Integer): HResult; stdcall;
    function GetPropName(lPropNum, lPropAlias: Integer; var pbstrPropName: WideString): HResult; stdcall;
    function GetPropVal(lPropNum: Integer; var pvarPropVal: OleVariant): HResult; stdcall;
    function SetPropVal(lPropNum: Integer; var varPropVal: OleVariant): HResult; stdcall;
    function IsPropReadable(lPropNum: Integer; var pboolPropRead: Integer): HResult; stdcall;
    function IsPropWritable(lPropNum: Integer; var pboolPropWrite: Integer): HResult; stdcall;
    function GetNMethods(var plMethods: Integer): HResult; stdcall;
    function FindMethod(const bstrMethodName: WideString; var plMethodNum: Integer): HResult; stdcall;
    function GetMethodName(lMethodNum, lMethodAlias: Integer; var pbstrMethodName: WideString): HResult; stdcall;
    function GetNParams(lMethodNum: Integer; var plParams: Integer): HResult; stdcall;
    function GetParamDefValue(lMethodNum, lParamNum: Integer; var pvarParamDefValue: OleVariant): HResult; stdcall;
    function HasRetVal(lMethodNum: Integer; var pboolRetValue: Integer): HResult; stdcall;
    function CallAsProc(lMethodNum: Integer; var paParams: PSafeArray{(OleVariant)}): HResult; stdcall;
    function CallAsFunc(lMethodNum: Integer; var pvarRetValue: OleVariant; var paParams: PSafeArray{(OleVariant)}): HResult; stdcall;
end;

implementation

uses Errors, CRC32, DIUCL;
(*
    DIUCL    UCL

UCL is Copyright (c) 1996-2002 Markus Franz Xaver Johannes Oberhumer
All Rights Reserved.
(c)Markus F.X.J. Oberhumer
<markus@oberhumer.com>
http://www.oberhumer.com/opensource/ucl/

DIUCL is Copyright (c) 2003 Ralf Junker - The Delphi Inspiration
All Rights Reserved.
Ralf Junker - The Delphi Inspiration
<delphi@zeitungsjunge.de>
http://www.zeitungsjunge.de/delphi/
*)


//   ,        1  7.7
//
//1  " "    0,      1..31
//     "   "
//( , 1   ""     ,     )
//
//     256     192,
//  48   .
//
//        : 256^18 -> 192^19
//..,   : 18 ""   19 192-
const
  convBlockLength256 = 18;
  convBlockLength192 = 19; //= Ceil(convBlockLength256*ln(256)/ln(192))

function Convert256to192(const Src: String): String;
var Length256, Length192, BlockCount: Cardinal;
var XLATTable: array [0..1023] of Word; //       192
begin
  Length256:= Length(Src);
  Length192:= Length256  + (Length256 * (convBlockLength192 - convBlockLength256) + convBlockLength256 - 1) div convBlockLength256;
  BlockCount:= (Length256 + convBlockLength256 - 1) div convBlockLength256;
  SetString(Result, nil, Length192 + convBlockLength192 + convBlockLength256); //    
  if BlockCount > 0 then
  asm
    PUSH ESI
    PUSH EDI
    PUSH EBX

//    
    CLD
    MOV ESI, Src
    MOV EDI, @Result
    MOV EDI, [EDI]
    ADD EDI, Length192
    ADD EDI, convBlockLength192
    SUB EDI, Length256
    MOV EDX, EDI
    MOV ECX, Length256
    REP MOVSB
    SUB EAX, EAX
    MOV ECX, convBlockLength256
    REP STOSB

// XLAT-
    STD
    MOV EDI, EBP
    ADD EDI, OFFSET XLATTable+767*2
    MOV AX, $80FF
@l0:STOSW
    SUB AH, $40
    JNC @l0
    MOV AH, $80
    SUB AL, $01
    JNC @l0

// 
    MOV EDI, EDX
    DEC EDI
    MOV EBX, @Result
    MOV EBX, [EBX]
@l1:
      MOV DH, convBlockLength192
      MOV DL, convBlockLength256
@l2:
        MOV CL, DL
        ADD EDI, ECX
        SUB EAX, EAX
        REPE SCASB
        MOV DL, CL
        JE @l4
        INC CL
        INC DL
        INC EDI
@l3:
          MOV AL, [EDI]
          MOV ESI, EAX                     //  
          SHR SI, 6                        //  
          SHL SI, 1                        // DIV 192
          MOV AH, AL                       //
          AND AX, $3F00                    //
          ADD AX, EBP+OFFSET XLATTable+ESI //
          STOSB
          LOOP @l3
@l4:
        ADD AX, $3000
        MOV [EBX], AH
        INC EBX
        DEC DH
        JNZ @l2

      ADD EDI, convBlockLength256
      DEC [BlockCount]
      JNZ @l1

    CLD
    POP EBX
    POP EDI
    POP ESI
  end;
  SetLength(Result, Length192);
end;

// 
function Convert192to256(const Src: String): String;
var Length256, Length192, BlockCount: Cardinal;
begin
  Length192:= Length(Src);
  Length256:= Length192  - (Length192 * (convBlockLength192 - convBlockLength256) + convBlockLength192 - 1) div convBlockLength192;
  BlockCount:= (Length192 + convBlockLength192 - 1) div convBlockLength192;
  SetString(Result, nil, Length192 + convBlockLength192 + convBlockLength256); //    
  if BlockCount > 0 then
  asm
    PUSH ESI
    PUSH EDI
    PUSH EBX

//    
    CLD
    MOV ESI, Src
    MOV EDI, @Result
    MOV EDI, [EDI]
    ADD EDI, convBlockLength256
    MOV EDX, EDI
    MOV ECX, Length192
@l0:
      LODSB
      SUB AL, $30
      STOSB
      LOOP @l0
    SUB EAX, EAX
    MOV ECX, convBlockLength192
    REP STOSB

// 
    STD
    MOV EDI, EDX
    DEC EDI
    MOV ESI, @Result
    MOV ESI, [ESI]
@l1:
      MOV DH, convBlockLength256
      MOV DL, convBlockLength192
@l2:
        MOV CL, DL
        ADD EDI, ECX
        SUB EAX, EAX
        REPE SCASB
        MOV DL, CL
        JE @l4
        INC CL
        INC DL
        INC EDI
@l3:
          MOV BH, AH
          SUB BL, BL
          SHR BX, 2
          MOV AL, [EDI]
          SUB AX, BX
          XCHG AH, AL
          STOSB
          LOOP @l3
@l4:
        MOV [ESI], AH
        INC ESI
        DEC DH
        JNZ @l2

      ADD EDI, convBlockLength192
      DEC [BlockCount]
      JNZ @l1

    CLD
    POP EBX
    POP EDI
    POP ESI
  end;
  SetLength(Result, Length256);
end;

function HexToBinStr(const S: String): String;
var Cnt: Integer;
begin
  Cnt:= Length(S) div 2;
  SetString(Result, nil, Cnt);
  Cnt:= HexToBin(@S[1], @Result[1], Cnt);
  if Cnt <> Length(Result) then SetLength(Result, Cnt)
end;

function BinToHexStr(const S: String): String;
begin
  SetString(Result, nil, Length(S)*2);
  BinToHex(@S[1], @Result[1], Length(S));
end;

function TAddInObject.LoadProperties: Boolean;
begin
   Result := True;
end;

procedure TAddInObject.SaveProperties;
begin

end;

function TAddInObject.GetNParam(var pArray : PSafeArray; lIndex: Integer ): OleVariant;
var varGet : OleVariant;
begin
   SafeArrayGetElement(pArray,lIndex,varGet);
   Result := varGet;
end;

procedure TAddInObject.PutNParam(var pArray: PSafeArray; lIndex: Integer; var varPut: OleVariant);
begin
  SafeArrayPutElement(pArray,lIndex,varPut);
end;

procedure TAddInObject.AddMsg(const Description: string; const wIcon: Word = ADDIN_E_ORDINARY);
var ExInfo: PExcepInfo;
begin
  New(ExInfo);
  ExInfo^.wCode:= wIcon;
  ExInfo^.sCode:= S_OK;
  ExInfo^.bstrSource:= 'AddIn.BinFile';
  ExInfo^.bstrDescription:= Description;
  pErrorLog.AddError(nil, ExInfo)
end;

{ IInitDone interface }

function TAddInObject.Init(const pConnection: IDispatch): HResult; stdcall;
var iRes : Integer;
begin
     pErrorLog := nil;
     pConnection.QueryInterface(IID_IErrorLog,pErrorLog);

     pEvent := nil;
     pConnection.QueryInterface(IID_IAsyncEvent,pEvent);

     pProfile := nil;
     iRes := pConnection.QueryInterface(IID_IPropertyProfile,pProfile);
     if (iRes = S_OK) then
        begin
             pProfile.RegisterProfileAs('AddIn.BinFile Profile Name');
             if (LoadProperties() <> True) then
               begin
                 Result := E_FAIL;
                 Exit;
               end;
        end;

     pStatusLine := nil;
     pConnection.QueryInterface(IID_IStatusLine,pStatusLine);

     FileName := '';
     FMode := -1;
     DebugMode := False;

     Result := S_OK;
end;

function TAddInObject.Done: HResult; stdcall;
begin
     if FMode <> -1 then try CloseFile(FHandle) except end;

     if (pErrorLog <> nil) then pErrorLog._Release();
     if (pEvent <> nil) then pEvent._Release();
     if (pProfile <> nil) then pProfile._Release();
     if (pStatusLine <> nil) then pStatusLine._Release();
     Result := S_OK;
end;

function TAddInObject.GetInfo(var pInfo: PSafeArray{(OleVariant)}): HResult; stdcall;
var  varInfo : OleVariant;
begin
     varInfo := '2000';
     PutNParam(pInfo,0,varInfo);
     Result := S_OK;
end;

{ ILanguageExtender interface }

function TAddInObject.RegisterExtensionAs(var bstrExtensionName: WideString): HResult; stdcall;
begin
     bstrExtensionName := 'BinFile';
     Result := S_OK;
end;

function TAddInObject.GetNProps(var plProps: Integer): HResult; stdcall;
begin
     plProps := Length(Props);
     Result := S_OK;
end;

function TAddInObject.FindProp(const bstrPropName: WideString; var plPropNum: Integer): HResult; stdcall;
var i: Integer;
begin
     plPropNum := -1;
     for i:= Low(Props) to High(Props) do
       if CompareText(bstrPropName, Props[i].Name) = 0 then begin
         plPropNum := i;
         Break
       end;
     if plPropNum = -1 Then Result := S_FALSE Else Result := S_OK;
end;

function TAddInObject.GetPropName(lPropNum, lPropAlias: Integer; var pbstrPropName: WideString): HResult; stdcall;
begin
     if (lPropNum < Low(Props)) or (lPropNum > High(Props)) then
       pbstrPropName := ''
     else
       pbstrPropName := Props[lPropNum].Name;
     if pbstrPropName = '' Then Result := S_FALSE Else Result := S_OK;
end;

function TAddInObject.IsPropReadable(lPropNum: Integer; var pboolPropRead: Integer): HResult; stdcall;
begin
  if Props[lPropNum].Readable then Result := S_OK else Result := S_FALSE;
end;

function TAddInObject.IsPropWritable(lPropNum: Integer; var pboolPropWrite: Integer): HResult; stdcall;
begin
  if Props[lPropNum].Writable then Result := S_OK else Result := S_FALSE;
end;

function TAddInObject.GetNMethods(var plMethods: Integer): HResult; stdcall;
begin
     plMethods := Length(Methods);
     Result := S_OK;
end;

function TAddInObject.FindMethod(const bstrMethodName: WideString; var plMethodNum: Integer): HResult; stdcall;
var i: Integer;
begin
     plMethodNum := -1;
     for i:= Low(Methods) to High(Methods) do
       if CompareText(bstrMethodName, Methods[i].Name) = 0 then begin
         plMethodNum := i;
         Break
       end;
     if plMethodNum = -1 then Result := S_FALSE else Result := S_OK;
end;

function TAddInObject.GetMethodName(lMethodNum, lMethodAlias: Integer; var pbstrMethodName: WideString): HResult; stdcall;
begin
     if (lMethodNum < Low(Methods)) or (lMethodNum > High(Methods)) then
       pbstrMethodName := ''
     else
       pbstrMethodName := Methods[lMethodNum].Name;
     if pbstrMethodName = '' then Result := S_FALSE else Result := S_OK;
end;

function TAddInObject.GetNParams(lMethodNum: Integer; var plParams: Integer): HResult; stdcall;
begin
     if (lMethodNum < Low(Methods)) or (lMethodNum > High(Methods)) then begin
       plParams := 0;
       Result := S_FALSE;
     end
     else begin
       plParams := Methods[lMethodNum].ParamCount;
       Result := S_OK;
     end;
end;

function TAddInObject.GetParamDefValue(lMethodNum, lParamNum: Integer; var pvarParamDefValue: OleVariant): HResult; stdcall;
begin
     VarClear(pvarParamDefValue);
     Result := S_FALSE;
     if (lMethodNum >= Low(Methods)) and (lMethodNum <= High(Methods))
     and (lParamNum < Methods[lMethodNum].ParamCount)
     and (Methods[lMethodNum].Params[lParamNum].HasDefValue <> dvNo) then begin
       Result := S_OK;
       case Methods[lMethodNum].Params[lParamNum].HasDefValue of
         dvInt: pvarParamDefValue:= Methods[lMethodNum].Params[lParamNum].DefInt;
         dvReal: pvarParamDefValue:= Methods[lMethodNum].Params[lParamNum].DefReal;
         dvStr: pvarParamDefValue:= Methods[lMethodNum].Params[lParamNum].DefStr;
       end;
     end;
end;

function TAddInObject.HasRetVal(lMethodNum: Integer; var pboolRetValue: Integer): HResult; stdcall;
begin
     Result := S_OK;
     pboolRetValue := 0;
     if (lMethodNum < Low(Methods)) or (lMethodNum > High(Methods)) then begin
       Result := S_FALSE;
     end
     else begin
       if Methods[lMethodNum].HasRetValue then pboolRetValue := 1
     end
end;

function TAddInObject.CallAsProc(lMethodNum: Integer; var paParams: PSafeArray{(OleVariant)}): HResult; stdcall;
var RetValue: OleVariant;
begin
    Result := CallAsFunc(lMethodNum, RetValue, paParams)
end;

function TAddInObject.GetPropVal(lPropNum: Integer; var pvarPropVal: OleVariant): HResult; stdcall;
var ExpectedError: Integer; Name: String;
begin
  Result := S_OK;
  //     ErrorCode
  //ErrorCode:= ercNoError;
  ExpectedError:= ercGeneralError;
  try
    Name := Props[lPropNum].Name;
    VarClear(pvarPropVal);

//##################################################
    if Name = strFileName then pvarPropVal := FileName

//##################################################
    else if Name = strEOF then begin
     if ((FMode = fmOpenRead) or (FMode = fmOpenReadWrite)) and (not EOF(FHandle)) then
       pvarPropVal := 0
     else
       pvarPropVal := 1
    end

//##################################################
    else if Name = strSize then begin
     if FMode = -1 then pvarPropVal := 0 else pvarPropVal := FileSize(FHandle);
    end

//##################################################
    else if Name = strPos then begin
     if FMode = -1 then pvarPropVal := 0 else pvarPropVal := FilePos(FHandle);
    end

//##################################################
    else if Name = strDebugMode then begin
     if DebugMode then pvarPropVal := 1 else pvarPropVal := 0;
    end

//##################################################
    else if Name = strErrorCode then pvarPropVal := ErrorCode

//##################################################
    else if Name = strErrorDescr then pvarPropVal := GetErrorDescription(ErrorCode)

//##################################################
    else begin
     Result := S_FALSE;
    end;

  except
    Result := E_FAIL;
    if ExpectedError = ercNoError then ErrorCode:= ercGeneralError else ErrorCode:= ExpectedError;
    if DebugMode then AddMsg(GetErrorDescription(ErrorCode), ADDIN_E_ATTENTION);
    exit;
  end;
end;

function TAddInObject.SetPropVal(lPropNum: Integer; var varPropVal: OleVariant): HResult; stdcall;
var ExpectedError: Integer; Name: String;
begin
  Result := S_OK;
  ErrorCode:= ercNoError;
  ExpectedError:= ercGeneralError;
  try
     Name := Props[lPropNum].Name;

//##################################################
     if Name = strPos then begin
       ExpectedError:= ercSeekError;
       if FMode <> -1 then Seek(FHandle, Integer(varPropVal))
     end

//##################################################
     else if Name = strDebugMode then begin
       DebugMode := (Integer(varPropVal) <> 0)
     end

//##################################################
     else begin
       Result := S_FALSE;
     end;

  except
    Result := E_FAIL;
    if ExpectedError = ercNoError then ErrorCode:= ercGeneralError else ErrorCode:= ExpectedError;
    if DebugMode then AddMsg(GetErrorDescription(ErrorCode), ADDIN_E_ATTENTION);
  end;
end;

const
  CurrencyFormat1C: TFormatSettings = (CurrencyFormat: 1; NegCurrFormat: 8; ThousandSeparator: #0; DecimalSeparator: '.'; CurrencyDecimals: 4);

function TAddInObject.CallAsFunc(lMethodNum: Integer; var pvarRetValue: OleVariant; var paParams: PSafeArray{(OleVariant)}): HResult; stdcall;
var ExpectedError: Integer; Name: String;
    S, S1, S2: String;
    N: Int64;
    Cnt: Integer;
    Sgl: Single;
    Dbl: Double;
    Param: OleVariant;
    Curr: array[0..0] of Currency;
    CRC32: Cardinal;
begin
  Result := S_OK;
  ErrorCode:= ercNoError;
  ExpectedError:= ercGeneralError;
  try
    Name := Methods[lMethodNum].Name;
    VarClear(pvarRetValue);

//##################################################
    if Name = strCreate then begin
      ExpectedError:= ercParamError;
      FileName:= GetNParam(paParams, 0);
      FMode:= -1;

      ExpectedError:= ercCreateError;
      AssignFile(FHandle, FileName);
      FileMode:= fmOpenWrite;
      Rewrite(FHandle,1);
      FMode:= fmOpenWrite;
    end

//##################################################
    else if Name = strOpen then begin
      ExpectedError:= ercParamError;
      FileName:= GetNParam(paParams, 0);
      Cnt:= GetNParam(paParams, 1);
      FMode:= -1;

      ExpectedError:= ercOpenError;
      AssignFile(FHandle, FileName);
      FileMode:= Cnt;
      Reset(FHandle,1);
      FMode:= Cnt;
    end

//##################################################
    else if Name = strClose then begin
      ExpectedError:= ercCloseError;
      if FMode <> -1 then CloseFile(FHandle);
      FMode:= -1;
    end

//##################################################
    else if Name = strWrite then begin
      ExpectedError:= ercParamError;
      S:= GetNParam(paParams, 0);
      Cnt:= GetNParam(paParams, 1);
      N:= GetNParam(paParams, 2);

      ExpectedError:= ercWriteModeError;
      if (FMode <> fmOpenWrite) and (FMode <> fmOpenReadWrite) then raise Exception.Create('');

      ExpectedError:= ercWriteError;
      if Cnt = -1 then Cnt:= Length(S) else if Cnt > Length(S) then S:= S + StringOfChar(Chr(N), Cnt-Length(S));
      BlockWrite(FHandle, S[1], Cnt);
    end

//##################################################
    else if Name = strWriteBin then begin
      ExpectedError:= ercParamError;
      N:= GetNParam(paParams, 0);
      Cnt:= GetNParam(paParams, 1);
      if Cnt > 4 then raise Exception.Create('');

      ExpectedError:= ercWriteModeError;
      if (FMode <> fmOpenWrite) and (FMode <> fmOpenReadWrite) then raise Exception.Create('');

      ExpectedError:= ercWriteError;
      if Cnt > 0 then  BlockWrite(FHandle, N, Cnt);
    end

//##################################################
    else if Name = strWriteSingle then begin
      ExpectedError:= ercParamError;
      Sgl:= GetNParam(paParams, 0);

      ExpectedError:= ercWriteModeError;
      if (FMode <> fmOpenWrite) and (FMode <> fmOpenReadWrite) then raise Exception.Create('');

      ExpectedError:= ercWriteError;
      BlockWrite(FHandle, Sgl, SizeOf(Sgl));
    end

//##################################################
    else if Name = strWriteDouble then begin
      ExpectedError:= ercParamError;
      Dbl:= GetNParam(paParams, 0);

      ExpectedError:= ercWriteModeError;
      if (FMode <> fmOpenWrite) and (FMode <> fmOpenReadWrite) then raise Exception.Create('');

      ExpectedError:= ercWriteError;
      BlockWrite(FHandle, Dbl, SizeOf(Dbl));
    end

//##################################################
    else if Name = strWriteCurrency then begin
      ExpectedError:= ercParamError;
      Param:= GetNParam(paParams, 0);
      if not (VarIsType(Param, varOleStr) or VarIsType(Param, varStrArg) or VarIsType(Param, varString)) then
        Curr[0]:= Param
      else begin
        S:= StringReplace(Param, ',', '.', [rfReplaceAll]);
        Curr[0]:= StrToCurr(S, CurrencyFormat1C);
      end;

      ExpectedError:= ercWriteModeError;
      if (FMode <> fmOpenWrite) and (FMode <> fmOpenReadWrite) then raise Exception.Create('');

      ExpectedError:= ercWriteError;
      BlockWrite(FHandle, Curr, SizeOf(Currency))
    end

//##################################################
    else if Name = strWriteHex then begin
      ExpectedError:= ercParamError;
      S:= GetNParam(paParams, 0);

      ExpectedError:= ercWriteModeError;
      if (FMode <> fmOpenWrite) and (FMode <> fmOpenReadWrite) then raise Exception.Create('');

      ExpectedError:= ercGeneralError;
      S:= HexToBinStr(LowerCase(S));
      ExpectedError:= ercWriteError;
      BlockWrite(FHandle, S[1], Length(S));
    end

//##################################################
    else if Name = strRead then begin
      ExpectedError:= ercParamError;
      Cnt:= GetNParam(paParams, 0);
      S1:= GetNParam(paParams, 1);

      ExpectedError:= ercReadModeError;
      if (FMode <> fmOpenRead) and (FMode <> fmOpenReadWrite) then raise Exception.Create('');

      ExpectedError:= ercGeneralError;
      if (Cnt = -1) or (Cnt > FileSize(FHandle) - FilePos(FHandle)) then Cnt:= FileSize(FHandle) - FilePos(FHandle);
      SetString(S, nil, Cnt);

      ExpectedError:= ercReadError;
      BlockRead(FHandle, S[1], Cnt);

      ExpectedError:= ercGeneralError;
      S2:= '';
      for Cnt:= 1 to Length(S) do
        if S[Cnt] = #0 then S2:= S2 + S1 else S2:= S2 + S[Cnt];
      pvarRetValue:= S2;
    end

//##################################################
    else if Name = strReadBin then begin
      ExpectedError:= ercParamError;
      Cnt:= GetNParam(paParams, 0);
      if Cnt > 4 then raise Exception.Create('');

      ExpectedError:= ercReadModeError;
      if (FMode <> fmOpenRead) and (FMode <> fmOpenReadWrite) then raise Exception.Create('');

      ExpectedError:= ercReadError;
      if Cnt > 0 then BlockRead(FHandle, N, Cnt) else N:= 0;
      pvarRetValue:= Integer(N);
    end

//##################################################
    else if Name = strReadSingle then begin
      ExpectedError:= ercReadModeError;
      if (FMode <> fmOpenRead) and (FMode <> fmOpenReadWrite) then raise Exception.Create('');

      ExpectedError:= ercReadError;
      Sgl:= 0;
      BlockRead(FHandle, Sgl, SizeOf(Sgl));
      pvarRetValue:= Sgl;
    end

//##################################################
    else if Name = strReadDouble then begin
      ExpectedError:= ercReadModeError;
      if (FMode <> fmOpenRead) and (FMode <> fmOpenReadWrite) then raise Exception.Create('');

      ExpectedError:= ercReadError;
      Dbl:= 0;
      BlockRead(FHandle, Dbl, SizeOf(Dbl));
      pvarRetValue:= Dbl
    end

//##################################################
    else if Name = strReadCurrency then begin
      ExpectedError:= ercReadModeError;
      if (FMode <> fmOpenRead) and (FMode <> fmOpenReadWrite) then raise Exception.Create('');

      ExpectedError:= ercReadError;
      Curr[0]:= 0;
      BlockRead(FHandle, Curr, SizeOf(Currency));
      pvarRetValue:= CurrToStr(Curr[0], CurrencyFormat1C);
    end

//##################################################
    else if Name = strReadHex then begin
      ExpectedError:= ercParamError;
      Cnt:= GetNParam(paParams, 0);

      ExpectedError:= ercReadModeError;
      if (FMode <> fmOpenRead) and (FMode <> fmOpenReadWrite) then raise Exception.Create('');

      ExpectedError:= ercGeneralError;
      if (Cnt = -1) or (Cnt > FileSize(FHandle) - FilePos(FHandle)) then Cnt:= FileSize(FHandle) - FilePos(FHandle);
      SetString(S, nil, Cnt);

      ExpectedError:= ercReadError;
      BlockRead(FHandle, S[1], Cnt);
      ExpectedError:= ercGeneralError;
      pvarRetValue:= BinToHexStr(S);
    end

//##################################################
    else if Name = strStrToHex then begin
      ExpectedError:= ercParamError;
      S:= GetNParam(paParams, 0);
      ExpectedError:= ercGeneralError;
      pvarRetValue:= BinToHexStr(S);
    end

//##################################################
    else if Name = strHexToStr then begin
      ExpectedError:= ercParamError;
      S:= GetNParam(paParams, 0);
      ExpectedError:= ercGeneralError;
      pvarRetValue:= HexToBinStr(LowerCase(S));
    end

//##################################################
    else if Name = strCompress then begin
      ExpectedError:= ercParamError;
      S:= GetNParam(paParams, 0);

      if S <> '' then begin //   ,  
        //  HEX-
        ExpectedError:= ercGeneralError;
        S:= HexToBinStr(LowerCase(S));

        //
        ExpectedError:= ercCompressError;
        S:= UclCompressStrA(S);

        //   CRC32
        ExpectedError:= ercCRC32CalcError;
        CRC32:= GetStringCRC32(S);
        Cnt:= Length(S);
        SetLength(S, Cnt+4);
        S[Cnt+1]:= Char(CRC32 and $FF);
        S[Cnt+2]:= Char(CRC32 shr 8 and $FF);
        S[Cnt+3]:= Char(CRC32 shr 16 and $FF);
        S[Cnt+4]:= Char(CRC32 shr 24 and $FF);

        //     1 
        ExpectedError:= ercConvertError;
        S:= Convert256to192(S);
      end;
      pvarRetValue:= S;
    end

//##################################################
    else if Name = strDecompress then begin
      ExpectedError:= ercParamError;
      S:= GetNParam(paParams, 0);

      if S <> '' then begin //   ,  
        //   1- 
        ExpectedError:= ercDeconvertError;
        S:= Convert192to256(S);

        // CRC32
        ExpectedError:= ercCRC32CalcError;
        CRC32:= GetStringCRC32(S);
        ExpectedError:= ercCRC32Error;
        if CRC32 <> 0 then raise Exception.Create('');
        if Length(S) <= 4 then SetLength(S, 0) else SetLength(S, Length(S)-4);

        //
        ExpectedError:= ercDecompressError;
        S:= UclDecompressStrA(S);

        //  HEX-
        ExpectedError:= ercGeneralError;
        S:= BinToHexStr(S);
      end;
      pvarRetValue:= S;
    end

//##################################################
    else CallAsFunc := Result;

  except
    Result := E_FAIL;
    if ExpectedError = ercNoError then ErrorCode:= ercGeneralError else ErrorCode:= ExpectedError;
    if DebugMode then AddMsg(GetErrorDescription(ErrorCode), ADDIN_E_ATTENTION);
  end;
end;

initialization
  ComServer.SetServerName('AddIn');
  TComObjectFactory.Create(ComServer,TAddInObject,CLSID_AddInObject,
                   'BinFile','V7 AddIn 2.0',ciMultiInstance);
end.
