// An example program that uses "macro (ie, persistent) scripts".

#define _WIN32_WINNT 0x0400
#include <windows.h>
#include <initguid.h>
#include <commctrl.h>
#include <objbase.h>
#include <activscp.h>
#include <tchar.h>
#include "IActiveScriptSite.h"
#include "resource.h"
#include "extern.h"


// This app's handle
HINSTANCE			InstanceHandle;

// Main window of our app
HWND				MainWindow;

// For running a script
static HANDLE			NotifySignal[2];
static IActiveScript	*EngineActiveScript;
static HANDLE			ThreadHandle;
static unsigned char	ScriptRunning;

// This is our "macro script" containing a SayHello subroutine that is
// going to be called by our VBscript[] script. NOTE: It must be UNICODE text
// format (ie, wchar_t, not char) because the script engine's
// ParseScriptText expects that. All this script does is display
// a "Hello world" messagebox
static const wchar_t	VBmacro[] = L"Sub HelloWorld\r\nMsgBox \"Hello world\"\r\nEnd Sub";

// This is our VBscript that we will run. For the sake of simplicity,
// we hard-code it inside of our app. All this script does is call the
// SayHello subroutine
static const wchar_t	VBscript[] = L"HelloWorld";

// For getting the engine's GUID from registry
static const TCHAR		CLSIDStr[] = _T("CLSID");
static const TCHAR		ScriptEngineStr[] = _T("ScriptEngine");

// Error message box
static const TCHAR		ErrorStr[] = _T("Error");
static const wchar_t	ErrorStrWide[] = L"Error";







/********************* display_sys_error() ********************
 * Displays a messagebox for the passed OS error number.
 *
 * NOTE: If passed a 0, this calls GetLastError().
 */

void display_sys_error(DWORD err)
{
	TCHAR	buffer[160];

	if (!err) err = GetLastError();		// If passed a 0, caller wants us to call GetLastError(). Do it FIRST!
	buffer[0] = 0;
	FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM, 0, err, MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), &buffer[0], 160, 0);
	MessageBox(MainWindow, &buffer[0], &ErrorStr[0], MB_OK);
}





/********************** display_COM_error() ********************
 * Displays a messagebox for a COM error.
 *
 * msg =		Format string for sprintf().
 * hr =			COM error number.
 *
 * NOTE: Total size of error msg must be < 256 TCHARs.
 */

void display_COM_error(LPCTSTR msg, HRESULT hr)
{
	TCHAR			buffer[256];

	wsprintf(&buffer[0], msg, hr);
	MessageBox(MainWindow, &buffer[0], &ErrorStr[0], MB_OK|MB_ICONEXCLAMATION);
}





/************************ getEngineGuid() ***********************
 * Gets the GUID of the script engine associated with the
 * specified filename extension.
 *
 * extension =	The filename extension.
 * guidBuffer =	Where to return the engine's GUID.
 *
 * RETURNS: S_OK if success, other for error.
 *
 * NOTE: Displays a messagebox for an error.
 */

static HRESULT getEngineGuid(LPCTSTR extension, GUID *guidBuffer)
{
	wchar_t		buffer[100];
	HKEY		hk;
	DWORD		size;
	HKEY		subKey;
	DWORD		type;

	// See if this file extension is associated with an ActiveX script engine
	if (!RegOpenKeyEx(HKEY_CLASSES_ROOT, extension, 0, KEY_QUERY_VALUE|KEY_READ, &hk))
	{
		type = REG_SZ;
		size = sizeof(buffer);
		size = RegQueryValueEx(hk, 0, 0, &type, (LPBYTE)&buffer[0], &size);
		RegCloseKey(hk);
		if (!size)
		{
			// The engine set an association. We got the Language string in buffer[]. Now
			// we can use it to look up the engine's GUID

			// Open HKEY_CLASSES_ROOT\{LanguageName}
again:			size = sizeof(buffer);
			if (!RegOpenKeyEx(HKEY_CLASSES_ROOT, (LPCTSTR)&buffer[0], 0, KEY_QUERY_VALUE|KEY_READ, &hk))
			{
				// Read the GUID (in string format) into buffer[] by querying the value of CLSID
				if (!RegOpenKeyEx(hk, &CLSIDStr[0], 0, KEY_QUERY_VALUE|KEY_READ, &subKey))
				{
					size = RegQueryValueExW(subKey, 0, 0, &type, (LPBYTE)&buffer[0], &size);
					RegCloseKey(subKey);
				}
				else if (extension)
				{
					// If an error, see if we have a "ScriptEngine" key under here that contains
					// the real language name
					if (!RegOpenKeyEx(hk, &ScriptEngineStr[0], 0, KEY_QUERY_VALUE|KEY_READ, &subKey))
					{
						size = RegQueryValueEx(subKey, 0, 0, &type, (LPBYTE)&buffer[0], &size);
						RegCloseKey(subKey);
						if (!size)
						{
							RegCloseKey(hk);
							extension = 0;
							goto again;
						}
					}
				}
			}

			RegCloseKey(hk);

			if (!size)
			{
				// Convert the GUID string to a GUID and put it in caller's guidBuffer
				if ((size = CLSIDFromString(&buffer[0], guidBuffer)))
					display_COM_error("Can't convert engine GUID: %08X", size);
				return(size);
			}
		}
	}

	MessageBox(0, "Can't get engine GUID from registry", &ErrorStr[0], MB_OK|MB_ICONEXCLAMATION);
	return(E_FAIL);
}





/************************** runScript() ***********************
 * Runs a script (on disk).
 *
 * args =	Contains the filename of the script to run, and
 *			the GUID of the script engine that runs the script.
 */

DWORD WINAPI runScript(void *args)
{
	register HRESULT	hr;
	IActiveScriptParse	*activeScriptParse;
	GUID				guidBuffer;

	// Each thread must initialize the COM system individually
	CoInitialize(0);

	// First, we do some initialization. This involves adding our VBS
	// "macro script" to the engine.

	// Find the script engine to use for files that end with a .VBS extension.
	// NOTE: Microsoft's VBscript engine sets up an association in the
	// registry for this extension.
	if (!(hr = getEngineGuid(".vbs", &guidBuffer)))
	{
		// Create an instance of the VBS engine, and get its IActiveScript object
		if ((hr = CoCreateInstance(&guidBuffer, 0, CLSCTX_ALL, &IID_IActiveScript, (void **)&EngineActiveScript)))
			PostMessage(MainWindow, WM_APP, (WPARAM)"Can't get engine's IActiveScript: %08X", hr);
		else
		{
			// Get the VBS engine's IActiveScriptParse object
			if ((hr = EngineActiveScript->lpVtbl->QueryInterface(EngineActiveScript, &IID_IActiveScriptParse, (void **)&activeScriptParse)))
				PostMessage(MainWindow, WM_APP, (WPARAM)"Can't get engine's IActiveScriptParse: %08X", hr);
			else
			{
				// Initialize the engine
				if ((hr = activeScriptParse->lpVtbl->InitNew(activeScriptParse)))
					PostMessage(MainWindow, WM_APP, (WPARAM)"Can't initialize engine : %08X", hr);
				else
				{
					// Give the engine our IActiveScriptSite object. If all goes well,
					// the engine will call its QueryInterface (which will AddRef it)
					if ((hr = EngineActiveScript->lpVtbl->SetScriptSite(EngineActiveScript, (IActiveScriptSite *)&MyActiveScriptSite)))
						PostMessage(MainWindow, WM_APP, (WPARAM)"Can't set our IScriptSite : %08X", hr);
					else
					{
						// Have the VBS engine parse our "macro script" and add it to its
						// internal list of scripts. NOTE: We specify SCRIPTTEXT_ISPERSISTENT so
						// this script stays loaded inside the engine until we Release the
						// engine. Also note that VBmacro[] contains a subroutine named SayHello,
						// so another script can call it, and we specify the SCRIPTTEXT_ISVISIBLE
						// flag so the other script can access it
						hr = activeScriptParse->lpVtbl->ParseScriptText(activeScriptParse, &VBmacro[0], 0, 0, 0, 0, 0, SCRIPTTEXT_ISPERSISTENT, 0, 0);

						// NOTE: If the script engine has a problem parsing/tokenizing the script, it will
						// have called our IActiveScriptSite's OnScriptError() to display an error msg, so
						// we don't need to do that here
						if (!hr)
						{
							// Let the main thread know that we're finished our initialization. We do
							// this by setting an event that the main thread is waiting upon.
							SetEvent(NotifySignal[1]);

							// Now we just wait for the main thread to signal us to run a script.
							for (;;)
							{
								// Wait for main thread to signal us to run a script, or abort.
								switch (WaitForMultipleObjects(2, &NotifySignal[0], 0, INFINITE))
								{
									// This signal means that main thread wants us to run a script
									case WAIT_OBJECT_0 + 0:
									{
										// Increment ScriptRunning to let main thread know a script is running
										++ScriptRunning;

										// Have the script engine parse our Vbscript and add it to its internal list
										// of scripts to run. NOTE: We do NOT specify SCRIPTTEXT_ISPERSISTENT so
										// this script will be unloaded when the engine goes back to INITIALIZED state
										hr = activeScriptParse->lpVtbl->ParseScriptText(activeScriptParse, &VBscript[0], 0, 0, 0, 0, 0, 0, 0, 0);
										if (!hr)
										{
											// Run all of the scripts that we added to the engine
											if ((hr = EngineActiveScript->lpVtbl->SetScriptState(EngineActiveScript, SCRIPTSTATE_CONNECTED)))
												PostMessage(MainWindow, WM_APP, (WPARAM)"Engine can't connect events: %08X", hr);
											else
											{
												// The above script has ended after SetScriptState returns. Now let's set
												// the engine state back to initialized to unload this script. VBmacro[]
												// remains still loaded.
												EngineActiveScript->lpVtbl->SetScriptState(EngineActiveScript, SCRIPTSTATE_INITIALIZED);
											}
										}

										// Decrement ScriptRunning to let main thread know a script is not running
										--ScriptRunning;

										break;
									}

									// This signal means the main thread wants this thread to end
									case WAIT_OBJECT_0 + 1:
										goto out;
								}
							}
						}
					}
				}

				// Release script engine's IActiveScriptParse
out:			activeScriptParse->lpVtbl->Release(activeScriptParse);
			}

			// We're supposed to Close() the engine before we Release() the IActiveScript.
			// VBmacro[] is unloaded now.
			EngineActiveScript->lpVtbl->Close(EngineActiveScript);
			
			// Release script engine's IActiveScript
			EngineActiveScript->lpVtbl->Release(EngineActiveScript);
		}
	}

	CoUninitialize();

	CloseHandle(ThreadHandle);

	// Indicate to main thread that I'm done running
	ThreadHandle = 0;
	SetEvent(NotifySignal[1]);

	// Terminate this thread
	return(0);
}





/**************************** mainWndProc() *****************************
 * The message procedure for MainWindow. It is called by Windows whenever
 * there is a message for MainWindow to process.
 */

static long APIENTRY mainWndProc(HWND hwnd, UINT uMsg, UINT wParam, long lParam)
{
	switch(uMsg)
	{
		// ***************************************************************
		// ======================== Menu or Buttons ======================
		case WM_COMMAND:
		{
			switch (wParam)
			{
				// -------------------- MENU: File -> Run script
		        case IDM_FILE_RUNSCRIPT:
				{
					// Is a script already running? If so, don't resignal the thread.
					if (!ThreadHandle || ScriptRunning) break;

					// Let the script thread know that we want it to run a script
					SetEvent(NotifySignal[0]);

					break;	// End of IDM_FILE_RUNSCRIPT
				}
			}

			break;
		}

		// ******************************************************************
		case WM_APP:
		{
			// Our script thread posts a WM_APP if it needs us to display an error message.
			// wParam = A pointer to the string to display. If 0, then lParam is an error
			// number to be passed to display_sys_error().
			// lParam = The HRESULT. If 0, then wParam is an allocated WCHAR string which
			// we must free with GlobalFree()
			if (!wParam)
				display_sys_error((DWORD)lParam);
			else if (!lParam)
			{
				MessageBoxW(hwnd, (const WCHAR *)wParam, &ErrorStrWide[0], MB_OK|MB_ICONEXCLAMATION);
				GlobalFree((void *)wParam);
			}
			else
				display_COM_error((LPCTSTR)wParam, (HRESULT)lParam);

			return(0);
		}

		// ******************************************************************
		// ====================== Create main window ========================
		case WM_INITDIALOG:
		{
			register HICON	icon;

			// Load/set icon for System menu on the window. I put my icon
			// in this executable's resources. Note that windows frees this
			// when my window is closed
			if ((icon = LoadIcon(InstanceHandle,MAKEINTRESOURCE(IDI_MAIN_ICON))))
				SetClassLong(hwnd, GCL_HICON, (LONG)icon);
  
			return(1);
		}

		// ******************************************************************
		// =================== User wants to close window ===================
		case WM_CLOSE:
		{
			// Let the script thread know that we want it to abort
			SetEvent(NotifySignal[1]);

			// Is a script running?
			if (ScriptRunning)
			{
				// Abort the script by calling InterruptScriptThread. This
				// is one of the few IActiveScript functions we can call by
				// any thread
				EngineActiveScript->lpVtbl->InterruptScriptThread(EngineActiveScript, SCRIPTTHREADID_ALL, 0, 0);
			}

			// Wait for the thread to actually end. It clears ThreadHandle right before
			// it ends. NOTE: We really should have a critical section around this access to
			// ThreadHandle since the script thread also accesses it. It is left up to
			// you to implement this
			wParam = 0;
			while (ThreadHandle && ++wParam < 25) Sleep(100);
			if (ThreadHandle) TerminateThread(ThreadHandle, 0);

			// Close this window
			DestroyWindow(hwnd);

			return(1);
		}

		case WM_DESTROY:
		{
 			// Post the WM_QUIT message to quit the message loop in WinMain()
			PostQuitMessage(0);

			return(1);
		}
	}

	// Indicate that I didn't handle the msg
	return(0);
} 





/************************** WinMain() **************************
 * Our EXE's entry point. This is called by Windows when our
 * EXE first starts up.
 */

int APIENTRY WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
	register HRESULT	hr;

	// Save Instance handle which I need when opening the "Choose engine" dialog
	// box. That dialog is linked into this executable's resources, so I need to pass
	// this handle to DialogBox() when I open that dialog
	InstanceHandle = hInstance;

	// Loads Windows common control's DLL
	InitCommonControls();

	// MainWindow not open yet
	MainWindow = 0;

	// Initialize COM DLL
	if ((hr = CoInitializeEx(0, COINIT_MULTITHREADED)))
		display_COM_error("Failed to initialize COM: %08X", hr);
	else
	{
		// Initialize MyRealIActiveScriptSite object
		initIActiveScriptSiteObject();

		// Get some event signals we need for communication with our script thread
		NotifySignal[1] = 0;
		if (!(NotifySignal[0] = CreateEvent(0, 0, 0, 0)) ||
			!(NotifySignal[1] = CreateEvent(0, 0, 0, 0)))
		{
			display_sys_error(0);
		}
		else
		{
			// Script not running yet
			ScriptRunning = 0;

			// Create Main window
			if (!(MainWindow = CreateDialog(InstanceHandle, MAKEINTRESOURCE(IDD_MAINWINDOW), 0, mainWndProc)))
				display_sys_error(0);
			else
			{
				MSG		msg;

				// Show the window with default size/position
				ShowWindow(MainWindow, SW_SHOWDEFAULT);
				UpdateWindow(MainWindow);

				// Create a secondary thread to run the script. It will do the
				// actual call to the engine to run the script while we just
				// manage the user interface
				if (!(ThreadHandle = CreateThread(0, 0, runScript, 0, 0, &msg.wParam)))
					display_sys_error(0);
				else
				{
					// Wait for the thread to finish its initialization
					WaitForSingleObject(NotifySignal[1], INFINITE);

					// Here's our message loop which is executed until the user closes
					// down our MainWindow
					while (GetMessage(&msg, 0, 0, 0) == 1)
					{
						TranslateMessage(&msg);
						DispatchMessage(&msg);
					}
				}
			}
		}

		// Free the signals
		if (NotifySignal[0]) CloseHandle(NotifySignal[0]);
		if (NotifySignal[1]) CloseHandle(NotifySignal[1]);

		// Allow our IActiveScriptSite to free any resource
		MyActiveScriptSite.site.lpVtbl->Release((IActiveScriptSite *)&MyActiveScriptSite);
	}

	// Free COM
	CoUninitialize();

	return(0);
}
