// The VTable and functions for our ISub2 sub-object of IMultInterface.

#include <windows.h>
#include <objbase.h>
#include <stddef.h>
#include "IMultInterface2.h"

// In our .H file, we use a macro which defines our ISub2 struct
// as so:
//
// typedef struct {
//    ISub2Vtbl  *lpVtbl;
// } ISub2;
//
// In other words, the .H file defines our ISub2 to have nothing
// but a pointer to its VTable. And of course, every COM object must
// start with a pointer to its VTable.
//
// But we actually want to add some more members to our ISub2.
// We just don't want any app to be able to know about, and directly
// access, those members. So here we'll define a MyRealISub2 that
// contains those extra members. The app doesn't know that we're
// really allocating and giving it a MyRealISub2 object. We'll
// lie and tell it we're giving a plain old ISub2. That's ok
// because a MyRealISub2 starts with the same VTable pointer.
//
// We add a DWORD variable that stores a value an app changes
// with the Increment and Decrement functions, and retrieves
// with the GetValue function.
//
// Because this is a sub-object of our IMultInterface, we need to
// save a pointer to IMultInterface's base object (ie, its
// IBase sub-object). That way we can call the base object's
// QueryInterface, AddRef, and Release when we need to
typedef struct {
	ISub2Vtbl		*lpVtbl;
#ifdef ISub2_NOPERSISTDATA
	long			count;
#endif
	long			value;
	IBase			*base;
} MyRealISub2;

// Here are ISub2's functions.
//
// Every COM object's interface must have the 3 functions QueryInterface(),
// AddRef(), and Release().
//
// I also chose to add 3 extra functions to ISub2.

// ISub2's QueryInterface()
static HRESULT STDMETHODCALLTYPE QueryInterface(ISub2 *this, REFIID vTableGuid, void **ppv)
{
	// Because ISub2 is a sub-object of one of our other COM objects (in the case, it's
	// a sub-object of our IMultInterface) then an app is allowed to call our ISub2's
	// QueryInterface to ask for a pointer not only to our ISub2 object, but also a
	// pointer to its base object, or any other sub-objects for that particular
	// IMultInterface.
	//
	// Because the base object's QueryInterface() already does all of this, the easiest
	// thing to do is somehow get a pointer to this base object, substitute it for the
	// "this" argument, and just call that base object's QueryInterface().
	//
	// When allocISub2() created this MyRealISub2, it saved a pointer to its
	// base object in the "base" member
	return(IBase_QueryInterface(((MyRealISub2 *)this)->base, vTableGuid, ppv));
}

// ISub2's AddRef()
static ULONG STDMETHODCALLTYPE AddRef(ISub2 *this)
{
#ifdef IEXTRA3_NOPERSISTDATA
	++(((MyRealISub2 *)this)->count);
#endif

	// Because ISub2 is a sub-object of our IMultInterface, then when an app
	// calls our ISub2's AddRef, we must also call its base object's AddRef to
	// increment the reference count. After all, we can't delete the base
	// object without first deleting all other sub-objects too. So to prevent
	// this from happening, we make sure that we increment the base reference
	// count every time an app AddRef()'s an ISub2.
	//
	// The easiest thing to do is somehow get a pointer to this ISub2's base
	// object, substitute it for the "this" argument, and just call that
	// base object's AddRef().
	//
	// When allocISub2() created this MyRealISub2, it saved a pointer to its
	// base object in the "base" member
	return(IBase_AddRef(((MyRealISub2 *)this)->base));
}

// ISub2's Release()
static ULONG STDMETHODCALLTYPE Release(ISub2 *this)
{
	register IBase		*base;

	base = ((MyRealISub2 *)this)->base;

// If you want persistent data in the ISub2, then you should forget
// about maintaining a separate reference count for it (and doing the
// code below. Instead, we'll GlobalFree the ISub2 only in
// IBase's Release (right before the whole IMultInterface itself is
// being GlobalFree'ed). In other words, once an app gets an ISub2,
// we'll keep that same object in MyRealIAudioObject->sub1 until
// MyReaIMultInterface itself is GlobalFree'ed
#ifdef ISub2_NOPERSISTDATA
	// If the app is done with this ISub2, we can delete it
	if (--(((MyRealISub2 *)this)->count) == 0)
	{	
		GlobalFree(this);

		// Make sure we zero the IMultInterface's "sub1" member. So if the
		// app asks for an ISub2 object again, we'll create one again.
		// NOTE: This means that an ISub2's private data is not persistent.
		// In other words, assume an app does a IBase->QueryInterface
		// to get an ISub2. Then the app calls ISub2->Increment to set the
		// value member to 1. Now, the app does a Release on the ISub2. We
		// therefore GlobalFree that ISub2 and zero MyReaIMultInterface->sub1.		
		// Then, the app does a IBase->QueryInterface to get an ISub2
		// again. This will be a different ISub2, and therefore its value
		// member will be initialized to 0 by allocISub2(). If the app does
		// a GetValue(), it will receive back the value 0; not 1.
		base->sub1 = 0;
	}
#endif

	// See AddRef() remarks above
	return(IBase_Release(base));
}

// ================== The following are my own extra functions added to ISub2

// ISub2's Increment(). This increments ISub2's "value" member
static HRESULT STDMETHODCALLTYPE Increment(ISub2 *this)
{
	++((MyRealISub2 *)this)->value;
	return(NOERROR);
}

// ISub2's Decrement(). This decrements ISub2's "value" member
static HRESULT STDMETHODCALLTYPE Decrement(ISub2 *this)
{
	--((MyRealISub2 *)this)->value;
	return(NOERROR);
}

// ISub2's GetValue(). This retrieves ISub2's "value" member
static HRESULT STDMETHODCALLTYPE GetValue(ISub2 *this, long *val)
{
	*val = ((MyRealISub2 *)this)->value;
	return(NOERROR);
}

// Here's ISub2's VTable. It never changes so we can declare it
// static
static const ISub2Vtbl ISub2_Vtbl = {QueryInterface,
AddRef,
Release,
Increment,
Decrement,
GetValue};

// This is just a helper function that allocates/initializes an ISub2.
ISub2 * allocISub2(IBase *base)
{
	register MyRealISub2	*sub;

	// Allocate an ISub2 (actually a MyRealISub2) object
	if ((sub = (MyRealISub2 *)GlobalAlloc(GMEM_FIXED, sizeof(MyRealISub2))))
	{
		// Set the VTable for the ISub2
		sub->lpVtbl = (ISub2Vtbl *)&ISub2_Vtbl;

		// Store the pointer to the IBase, so we can get it when we need it
		sub->base = base;

		// Initialize other private data members
#ifdef ISub2_NOPERSISTDATA
		sub->count = 0;
#endif
		sub->value = 0;
	}

	return((ISub2 *)sub);
}