// An example of a JScript calling a VBScript (ie, tying two engines
// together).

#include <windows.h>
#include <stdio.h>
#include <initguid.h>
#include <objbase.h>
#include <activscp.h>
#include <tchar.h>
#include "IActiveScriptSite.h"
#include "extern.h"

// IActiveScript for VBScript engine
IActiveScript			*VBActiveScript;

// IActiveScript for JScript engine
IActiveScript			*JActiveScript;

// The named item we add to the VB engine
const wchar_t			VBName[] = L"VB";

// The named item we add to the JScript engine
const wchar_t			JName[] = L"JSCRIPT";

// This is our VB script containing a SayHello subroutine that our
// JScript will call. NOTE: It must be UNICODE text
// format (ie, wchar_t, not char) because the script engine's
// ParseScriptText expects that. All this script does is display
// a "Hello world" messagebox
static const wchar_t	VBscript[] = L"Sub SayHello\r\nMsgBox \"Hello World\"\r\nEnd Sub";

// This is our JScript. It calls the VB script's SayHello sub
static const wchar_t	Jscript[] = L"function main()\r\n {\r\n SayHello();\r\n }";

// For getting the engine's GUID from registry
static const TCHAR		CLSIDStr[] = _T("CLSID");
static const TCHAR		ScriptEngineStr[] = _T("ScriptEngine");

// Error message box
static const TCHAR		ErrorStr[] = _T("Error");





/********************** display_COM_error() ********************
 * Displays a messagebox for a COM error.
 *
 * msg =		Format string for sprintf().
 * hr =			COM error number.
 *
 * NOTE: Total size of error msg must be < 256 TCHARs.
 */

void display_COM_error(LPCTSTR msg, HRESULT hr)
{
	TCHAR			buffer[256];

	wsprintf(&buffer[0], msg, hr);
	MessageBox(0, &buffer[0], &ErrorStr[0], MB_OK|MB_ICONEXCLAMATION);
}





/************************ getEngineGuid() ***********************
 * Gets the GUID of the script engine associated with the
 * specified filename extension.
 *
 * extension =	The filename extension.
 * guidBuffer =	Where to return the engine's GUID.
 *
 * RETURNS: S_OK if success, other for error.
 *
 * NOTE: Displays a messagebox for an error.
 */

static HRESULT getEngineGuid(LPCTSTR extension, GUID *guidBuffer)
{
	wchar_t		buffer[100];
	HKEY		hk;
	DWORD		size;
	HKEY		subKey;
	DWORD		type;

	// See if this file extension is associated with an ActiveX script engine
	if (!RegOpenKeyEx(HKEY_CLASSES_ROOT, extension, 0, KEY_QUERY_VALUE|KEY_READ, &hk))
	{
		type = REG_SZ;
		size = sizeof(buffer);
		size = RegQueryValueEx(hk, 0, 0, &type, (LPBYTE)&buffer[0], &size);
		RegCloseKey(hk);
		if (!size)
		{
			// The engine set an association. We got the Language string in buffer[]. Now
			// we can use it to look up the engine's GUID

			// Open HKEY_CLASSES_ROOT\{LanguageName}
again:			size = sizeof(buffer);
			if (!RegOpenKeyEx(HKEY_CLASSES_ROOT, (LPCTSTR)&buffer[0], 0, KEY_QUERY_VALUE|KEY_READ, &hk))
			{
				// Read the GUID (in string format) into buffer[] by querying the value of CLSID
				if (!RegOpenKeyEx(hk, &CLSIDStr[0], 0, KEY_QUERY_VALUE|KEY_READ, &subKey))
				{
					size = RegQueryValueExW(subKey, 0, 0, &type, (LPBYTE)&buffer[0], &size);
					RegCloseKey(subKey);
				}
				else if (extension)
				{
					// If an error, see if we have a "ScriptEngine" key under here that contains
					// the real language name
					if (!RegOpenKeyEx(hk, &ScriptEngineStr[0], 0, KEY_QUERY_VALUE|KEY_READ, &subKey))
					{
						size = RegQueryValueEx(subKey, 0, 0, &type, (LPBYTE)&buffer[0], &size);
						RegCloseKey(subKey);
						if (!size)
						{
							RegCloseKey(hk);
							extension = 0;
							goto again;
						}
					}
				}
			}

			RegCloseKey(hk);

			if (!size)
			{
				// Convert the GUID string to a GUID and put it in caller's guidBuffer
				if ((size = CLSIDFromString(&buffer[0], guidBuffer)))
					display_COM_error("Can't convert engine GUID: %08X", size);
				return(size);
			}
		}
	}

	MessageBox(0, "Can't get engine GUID from registry", &ErrorStr[0], MB_OK|MB_ICONEXCLAMATION);
	return(E_FAIL);
}





/************************** runScript() ***********************
 * Runs our scripts.
 *
 * If an error, this displays a message box.
 */

void runScript(void)
{
	register HRESULT	hr;

	VBActiveScript = JActiveScript = 0;

	{
	GUID	guidBuffer;

	// Find the script engine to use for files that end with a .VBS extension.
	// NOTE: Microsoft's VBscript engine sets up an association in the
	// registry for this extension.
	if (!getEngineGuid(".vbs", &guidBuffer))
	{
		// Create an instance of the VB engine, and get its IActiveScript object
		if ((hr = CoCreateInstance(&guidBuffer, 0, CLSCTX_ALL, &IID_IActiveScript, (void **)&VBActiveScript)))
			display_COM_error("Can't get VB engine's IActiveScript: %08X", hr);
		else
		{
			// Find the script engine to use for files that end with a .JS extension.
			// NOTE: Microsoft's Jscript engine sets up an association in the
			// registry for this extension.
			if (!getEngineGuid(".js", &guidBuffer))
			{
				// Create an instance of the Jscript engine, and get its IActiveScript object
				if ((hr = CoCreateInstance(&guidBuffer, 0, CLSCTX_ALL, &IID_IActiveScript, (void **)&JActiveScript)))
					display_COM_error("Can't get Jscript engine's IActiveScript: %08X", hr);
			}
		}
	}
	}

	if (VBActiveScript && JActiveScript)
	{
		IActiveScriptParse	*vbActiveScriptParse;
		IActiveScriptParse	*jActiveScriptParse;

		// Get the VB engine's IActiveScriptParse object (which we can do from its
		// IActiveScript's QueryInterface since IActiveScriptParse is a
		// sub-object of the IActiveScript)
		if ((hr = VBActiveScript->lpVtbl->QueryInterface(VBActiveScript, &IID_IActiveScriptParse, (void **)&vbActiveScriptParse)))
			display_COM_error("Can't get VB engine's IActiveScriptParse: %08X", hr);
		else
		{
			// Get the JScript engine's IActiveScriptParse object
			if ((hr = JActiveScript->lpVtbl->QueryInterface(JActiveScript, &IID_IActiveScriptParse, (void **)&jActiveScriptParse)))
				display_COM_error("Can't get JScript engine's IActiveScriptParse: %08X", hr);
			else
			{
				// Initialize both engines. This just lets the engine internally
				// initialize some stuff in preparation of us adding scripts to it
				// for the first time
				if ((hr = vbActiveScriptParse->lpVtbl->InitNew(vbActiveScriptParse)) ||
				   (hr = jActiveScriptParse->lpVtbl->InitNew(jActiveScriptParse)))
				{
					display_COM_error("Can't initialize engine : %08X", hr);
				}
				else
				{
					// Give both engines our IActiveScriptSite object. If all goes well,
					// the engine will call its QueryInterface (which will AddRef it)
					if ((hr = VBActiveScript->lpVtbl->SetScriptSite(VBActiveScript, (IActiveScriptSite *)&MyActiveScriptSite)) ||
						(hr = JActiveScript->lpVtbl->SetScriptSite(JActiveScript, (IActiveScriptSite *)&MyActiveScriptSite)))
					{
						display_COM_error("Can't set our IScriptSite : %08X", hr);
					}
					else
					{
						// Create a named item in the VB engine. Our IActiveScriptSite's GetItemInfo() will associate
						// this named item with the JScript (ie, other) engine. In other words, when the VB engine
						// calls our GetItemInfo() asking for this named item, we will return the IDispatch for the
						// JScript engine's global named item. In this way, the VB engine can then use this IDispatch
						// to directly call any JScript function. Let's arbitrarily choose a name of "JScript". NOTE:
						// We specify ISVISIBLE so when we call GetScriptDispatch, we can get the IDispatch. We also
						// specify GLOBALMEMBERS so that, when a VB script calls a JScript function, it doesn't need
						// to include an object name of "JScript".
						//
						// NOTE: We don't actually use this named item in our example, because we don't have any
						// VBScript calling a JScript function. But if we did, we'd need it.
						if ((hr = VBActiveScript->lpVtbl->AddNamedItem(VBActiveScript, &JName[0], SCRIPTITEM_GLOBALMEMBERS|SCRIPTITEM_ISVISIBLE)) ||

							// Create a named item in the JScript engine. Our IActiveScriptSite's GetItemInfo() will associate
							// this named item with the VB (ie, other) engine. In other words, when the JScript engine
							// calls our GetItemInfo() asking for this named item, we will return the IDispatch for the
							// VB engine's global named item. In this way, the JScript engine can then use this IDispatch
							// to directly call any VB function. Let's arbitrarily choose a name of "VB".
							(hr = JActiveScript->lpVtbl->AddNamedItem(JActiveScript, &VBName[0], SCRIPTITEM_GLOBALMEMBERS|SCRIPTITEM_ISVISIBLE)))
						{
							display_COM_error("Can't create named item : %08X", hr);
						}
						else
						{
							// Have the script engine parse the script and add it to its internal
							// list of scripts to run
							if (!(hr = vbActiveScriptParse->lpVtbl->ParseScriptText(vbActiveScriptParse, &VBscript[0], 0, 0, 0, 0, 0, SCRIPTITEM_ISVISIBLE, 0, 0)) &&
								!(hr = jActiveScriptParse->lpVtbl->ParseScriptText(jActiveScriptParse, &Jscript[0], 0, 0, 0, 0, 0, SCRIPTITEM_ISVISIBLE, 0, 0)))

							// NOTE: If the script engine has a problem parsing/tokenizing the script, it will
							// have called our IActiveScriptSite's OnScriptError() to display an error msg, so
							// we don't need to do that here
							{
								// Set both engine's state to CONNECTED. Our above scripts will run now. But since
								// there is no code outside of any sub/function, no instructions actually get
								// executed.
								if ((hr = JActiveScript->lpVtbl->SetScriptState(JActiveScript, SCRIPTSTATE_CONNECTED)) ||
									(hr = VBActiveScript->lpVtbl->SetScriptState(VBActiveScript, SCRIPTSTATE_CONNECTED)))
								{
									display_COM_error("Engine can't connect events: %08X", hr);
								}
								else
								{
									IDispatch	*objPtr;

									// Get the IDispatch for the default "global named item"
									hr = JActiveScript->lpVtbl->GetScriptDispatch(JActiveScript, 0, &objPtr);
									if (hr)
										display_COM_error("Can't get our IDispatch : %08X", hr);
									else
									{
										DISPID		dispid;
										OLECHAR		*funcName;

										// Get the DISPID for the "main" sub
										funcName = (OLECHAR *)L"main";
										hr = objPtr->lpVtbl->GetIDsOfNames(objPtr, &IID_NULL, &funcName, 1, LOCALE_USER_DEFAULT, &dispid);
										if (hr)
											display_COM_error("Can't get our DISPID : %08X", hr);
										else
										{
											DISPPARAMS	dspp;
											VARIANT		ret;

											// Call main. Since main has no args passed to it, we don't have to do
											// any grotesque initialization of DISPPARAMS.
											ZeroMemory(&dspp, sizeof(DISPPARAMS));
											VariantInit(&ret);
											objPtr->lpVtbl->Invoke(objPtr, dispid, &IID_NULL, LOCALE_USER_DEFAULT, DISPATCH_METHOD, &dspp, &ret, 0, 0);
											VariantClear(&ret);
										}

										// Release the IDispatch now that we made the call
										objPtr->lpVtbl->Release(objPtr);
									}
								}
							}
						}
					}
				}

				// Release JScript engine's IActiveScriptParse
				jActiveScriptParse->lpVtbl->Release(jActiveScriptParse);
			}

			// Release VB engine's IActiveScriptParse
			vbActiveScriptParse->lpVtbl->Release(vbActiveScriptParse);
		}

		// We're supposed to Close() the engine before we Release() the IActiveScript
		VBActiveScript->lpVtbl->Close(VBActiveScript);

		// We're supposed to Close() the engine before we Release() the IActiveScript
		JActiveScript->lpVtbl->Close(JActiveScript);

		// Release Jscript engine's IActiveScript
		JActiveScript->lpVtbl->Release(JActiveScript);
	}

	// Release VB engine's IActiveScript
	if (VBActiveScript) VBActiveScript->lpVtbl->Release(VBActiveScript);
}





/************************** main() **************************
 * Our EXE's entry point. This is called by Windows when our
 * EXE first starts up.
 */

int main(int argc, char **argv)
{
	register HRESULT	hr;

	// Initialize COM DLL
	if ((hr = CoInitialize(0)))
		display_COM_error("Failed to initialize COM: %08X", hr);
	else
	{
		// Initialize MyRealIActiveScriptSite object
		initIActiveScriptSiteObject();

		// Run the scripts
		runScript();

		// Allow our IActiveScriptSite to free any resources
		MyActiveScriptSite.site.lpVtbl->Release((IActiveScriptSite *)&MyActiveScriptSite);
	}

	// Free COM
	CoUninitialize();

	return(0);
}
