// Main functions for an example program that loads and executes a script.

#define _WIN32_WINNT 0x0400
#include <windows.h>
#include <initguid.h>
#include <commctrl.h>
#include <objbase.h>
#include <activscp.h>
#include <tchar.h>
#include "TraceWin.h"
#include "IActiveScriptSite.h"
#include "resource.h"
#include "extern.h"

typedef struct {
	IActiveScript	*EngineActiveScript;
	HANDLE			ThreadHandle;
	TCHAR			Filename[MAX_PATH];
	GUID			Guid;
} MYARGS;

// This app's handle
HINSTANCE			InstanceHandle;

// Main window of our app
HWND				MainWindow;

// For running a script
MYARGS				MyArgs;

// The name of the function that is the entry point of any script we run
static const wchar_t	MainFunction[] = L"main";

// For script file dialog
static const TCHAR	Load[] = "Pick script to run (or Cancel to quit):";
static const TCHAR	ExtensionList[] = {'V','B','s','c','r','i','p','t',' ','(','*','.','v','b','s',')',0,
								'*','.','v','b','s',0,
								'J','S','c','r','i','p','t',' ','(','*','.','v','j','s',')',0,
								'*','.','j','v','s',0,
								'A','l','l',' ','f','i','l','e','s',' ','(','*','.','*',')',0,
								'*','.','*',0,0};

// For getting engine's GUID from registry
static const TCHAR		CLSIDStr[] = _T("CLSID");
static const TCHAR		ScriptEngineStr[] = _T("ScriptEngine");

// Error message box
static const TCHAR		ErrorStr[] = _T("Error");
static const wchar_t	ErrorStrWide[] = L"Error";






/************************* getLoadName() ***********************
 * Gets the user's choice of script filename, and copies it
 * to Filename[].
 *
 * RETURNS: 0 if success, or non-zero if cancel/failure.
 *
 * NOTE: Filename[] must be MAX_PATH size.
 */

static BOOL getLoadName(MYARGS *args)
{
	OPENFILENAME	ofn;

	// Clear out fields
	ZeroMemory(&ofn, sizeof(OPENFILENAME));

	// Store passed buffer
	ofn.lpstrFile = &args->Filename[0];
	args->Filename[0] = 0;
	ofn.nMaxFile = MAX_PATH;

	// Set size
	ofn.lStructSize = sizeof(OPENFILENAME);

	// Set extensions
	ofn.lpstrFilter = &ExtensionList[0];

	// Set title
	ofn.lpstrTitle = (LPTSTR)&Load[0];

	// Set flags
	ofn.Flags = OFN_PATHMUSTEXIST|OFN_FILEMUSTEXIST|OFN_LONGNAMES|OFN_EXPLORER|OFN_HIDEREADONLY;

	// Present the dialog and get user's selection
	if (GetOpenFileName(&ofn))
	{
		// Return OK
		return(0);
	}

	// Abort
	return(1);
}





/********************* display_sys_error() ********************
 * Displays a messagebox for the passed OS error number.
 *
 * NOTE: If passed a 0, this calls GetLastError().
 */

void display_sys_error(DWORD err)
{
	TCHAR	buffer[160];

	if (!err) err = GetLastError();		// If passed a 0, caller wants us to call GetLastError(). Do it FIRST!
	buffer[0] = 0;
	FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM, 0, err, MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), &buffer[0], 160, 0);
	MessageBox(MainWindow, &buffer[0], &ErrorStr[0], MB_OK);
}





/*********************** loadUnicodeScript() ********************
 * Reads a script off of disk, and copies it to a UNICODE
 * buffer.
 *
 * fn =		Filename of script.
 *
 * RETURNS: A pointer to the allocated UNICODE buffer if success,
 * or zero if failure.
 *
 * NOTE: Caller must GlobalFree() the returned buffer.
 *
 * Displays an error message if a failure.
 */

static OLECHAR * loadUnicodeScript(LPCTSTR fn)
{
	register OLECHAR	*script;
	register HANDLE		hfile;
	DWORD				error;

	// Assume no error
	error = 0;

	// Open the file
	if ((hfile = CreateFile(fn, GENERIC_READ, FILE_SHARE_READ, 0, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, 0)) != INVALID_HANDLE_VALUE)
	{
		DWORD	filesize;
		char	*psz;

		// Get a buffer to read in the file
		filesize = GetFileSize(hfile, 0);
		if ((psz = (char *)GlobalAlloc(GMEM_FIXED, filesize + 1)))
 		{
			DWORD	read;

			// Read in the file
			ReadFile(hfile, psz, filesize, &read, 0);

			// Get a buffer to convert to UNICODE (plus an extra wchar_t to nul-terminate it since
			// the engine's IActiveScriptParse ParseScriptText expects that)
			if ((script = (OLECHAR *)GlobalAlloc(GMEM_FIXED, (filesize + 1) * sizeof(OLECHAR))))
			{
				// Convert to UNICODE and nul-terminate
				MultiByteToWideChar(CP_ACP, 0, psz, filesize, script, filesize + 1);
				script[filesize] = 0;
			}
			else
				error = GetLastError();

			GlobalFree(psz);
		}
		else
			error = GetLastError();

		CloseHandle(hfile);
	}
	else
		error = GetLastError();

	if (error)
	{
		PostMessage(MainWindow, WM_APP, 0, error);
		script = 0;
	}

	return(script);
}





/********************** display_COM_error() ********************
 * Displays a messagebox for a COM error.
 *
 * msg =		Format string for sprintf().
 * hr =			COM error number.
 *
 * NOTE: Total size of error msg must be < 256 TCHARs.
 */

void display_COM_error(LPCTSTR msg, HRESULT hr)
{
	TCHAR			buffer[256];

	wsprintf(&buffer[0], msg, hr);
	MessageBox(MainWindow, &buffer[0], &ErrorStr[0], MB_OK|MB_ICONEXCLAMATION);
}





/************************** runScript() ***********************
 * Runs a script (on disk).
 *
 * args =	Contains the filename of the script to run, and
 *			the GUID of the script engine that runs the script.
 */

DWORD WINAPI runScript(MYARGS *args)
{
	register HRESULT	hr;
	IActiveScriptParse	*activeScriptParse;

	// Each thread must initialize the COM system individually
	CoInitialize(0);

	// Create an instance of the script engine, and get its IActiveScript object
	if ((hr = CoCreateInstance(&args->Guid, 0, CLSCTX_ALL, &IID_IActiveScript, (void **)&args->EngineActiveScript)))
		PostMessage(MainWindow, WM_APP, (WPARAM)"Can't get engine's IActiveScript: %08X", hr);
	else
	{
		// Get the script engine's IActiveScriptParse object (which we can do from its
		// IActiveScript's QueryInterface since IActiveScriptParse is a
		// sub-object of the IActiveScript)
		if ((hr = args->EngineActiveScript->lpVtbl->QueryInterface(args->EngineActiveScript, &IID_IActiveScriptParse, (void **)&activeScriptParse)))
			PostMessage(MainWindow, WM_APP, (WPARAM)"Can't get engine's IActiveScriptParse: %08X", hr);
		else
		{
			// Initialize the engine. This just lets the engine internally
			// initialize some stuff in preparation of us adding scripts to it
			// for the first time
			if ((hr = activeScriptParse->lpVtbl->InitNew(activeScriptParse)))
				PostMessage(MainWindow, WM_APP, (WPARAM)"Can't initialize engine : %08X", hr);
			else
			{
				// Give the engine our IActiveScriptSite object. If all goes well,
				// the engine will call its QueryInterface (which will AddRef it)
				if ((hr = args->EngineActiveScript->lpVtbl->SetScriptSite(args->EngineActiveScript, (IActiveScriptSite *)&MyActiveScriptSite)))
					PostMessage(MainWindow, WM_APP, (WPARAM)"Can't set our IScriptSite : %08X", hr);
				else
				{
					register LPOLESTR	str;

					// Load the script from disk. NOTE: We need to load it UNICODE for ParseScriptText()
					if (!(str = loadUnicodeScript(&args->Filename[0])))
						PostMessage(MainWindow, WM_APP, (WPARAM)"Can't load script from disk", E_FAIL);
					else
					{
						// Have the script engine parse it and add it to its internal list
						// of scripts to run. NOTE: We don't pass any object name, so this
						// script is put inside the "global or default object" and therefore
						// this script will be run as soon as we set the engine to the
						// connected state
						hr = activeScriptParse->lpVtbl->ParseScriptText(activeScriptParse, str, 0, 0, 0, 0, 0, 0, 0, 0);

						// We no longer need the loaded script
						GlobalFree(str);

						// NOTE: If the script engine has a problem parsing/tokenizing the script, it will
						// have called our IActiveScriptSite's OnScriptError() to display an error msg, so
						// we don't need to do that here
						if (!hr &&

							// Set engine's state to CONNECTED. NOTE: If we called the engine's AddNamedItem()
							// to add some objects, then the script engine will QueryInterface our
							// IActiveScriptSite for the needed IDispatch objects from us
							(hr = args->EngineActiveScript->lpVtbl->SetScriptState(args->EngineActiveScript, SCRIPTSTATE_CONNECTED)))
						{
							PostMessage(MainWindow, WM_APP, (WPARAM)"Engine can't connect events: %08X", hr);
						}
					}
				}
			}

			// Release script engine's IActiveScriptParse
			activeScriptParse->lpVtbl->Release(activeScriptParse);
		}

		// We're supposed to Close() the engine before we Release() the IActiveScript
		args->EngineActiveScript->lpVtbl->Close(args->EngineActiveScript);
		
		// Release script engine's IActiveScript
		args->EngineActiveScript->lpVtbl->Release(args->EngineActiveScript);
	}

	CoUninitialize();

	// NOTE: We really should have a critical section around this access to
	// ThreadHandle since the main thread also accesses it. It is left up to
	// you to implement this

	// Let main thread know that we're done (running the script)
	CloseHandle(args->ThreadHandle);
	args->ThreadHandle = 0;

	// Terminate this thread
	return(0);
}





/**************************** mainWndProc() *****************************
 * The message procedure for MainWindow. It is called by Windows whenever
 * there is a message for MainWindow to process.
 */

long APIENTRY mainWndProc(HWND hwnd, UINT uMsg, UINT wParam, long lParam)
{
	switch(uMsg)
	{
		// ***************************************************************
		// ======================== Menu or Buttons ======================
		case WM_COMMAND:
		{
			switch (wParam)
			{
				// -------------------- MENU: File -> Run script
		        case IDM_FILE_RUNSCRIPT:
				{
					// Is a script already running? If so, don't rerun the thread.
					// We do this because we have only one global MYARGS. If you
					// wanted to run multiple scripts simultaneously, then you'd
					// dynamically allocate each needed MYARGS here, adding a member
					// to it in order to link them all into one list
					if (MyArgs.ThreadHandle) break;

					// Get the filename of the script to run
					if (!getLoadName(&MyArgs))
					{
						LPCTSTR		extension;
						HKEY		hk;

						// Ok, let's see if we can find the appropriate script engine to use,
						// based upon the extension on the filename. Hopefully the script
						// engine associated itself with a particular file extension. If so,
						// then it should have set a registry key under HKEY_CLASSES_ROOT.

						// Isolate the extension on the filename
						extension = &MyArgs.Filename[0] + lstrlen(&MyArgs.Filename[0]);
						goto backup;
						while (*extension != '.')
						{
							if (*extension == '\\' || extension <= &MyArgs.Filename[0])
							{
								// No extension. We'll need to let the user pick out the engine
								goto choose;
							}

backup:						extension = CharPrev(&MyArgs.Filename[0], extension);
						}

						// See if the engine set a file association
						if (RegOpenKeyEx(HKEY_CLASSES_ROOT, extension, 0, KEY_QUERY_VALUE|KEY_READ, &hk))
						{
							// It didn't. We'll have to let the user pick a script engine, and
							// copy the GUID to MYARGS
choose:						lParam = chooseEngineDlg(&MyArgs.Guid);
						}
						else
						{
							HKEY		subKey;
							wchar_t		language[100];

							wParam = REG_SZ;
							lParam = sizeof(language);
							lParam = RegQueryValueEx(hk, 0, 0, &wParam, (LPBYTE)&language[0], &lParam);
							RegCloseKey(hk);
							if (lParam) goto choose;

							// The engine set an association. We got the Language string in language[]. Now
							// we can use it to look up the engine's GUID

							// Open HKEY_CLASSES_ROOT\{LanguageName}
again:						lParam = sizeof(language);
							if (!RegOpenKeyEx(HKEY_CLASSES_ROOT, (LPCTSTR)&language[0], 0, KEY_QUERY_VALUE|KEY_READ, &hk))
							{
								// Read the GUID (in string format) into language[] by querying the value of CLSID
								if (!RegOpenKeyEx(hk, &CLSIDStr[0], 0, KEY_QUERY_VALUE|KEY_READ, &subKey))
								{
									lParam = RegQueryValueExW(subKey, 0, 0, &wParam, (LPBYTE)&language[0], &lParam);
									RegCloseKey(subKey);
								}
								else if (extension)
								{
									// If an error, see if we have a "ScriptEngine" key under here that contains
									// the real language name
									if (!RegOpenKeyEx(hk, &ScriptEngineStr[0], 0, KEY_QUERY_VALUE|KEY_READ, &subKey))
									{
										lParam = RegQueryValueEx(subKey, 0, 0, &wParam, (LPBYTE)&language[0], &lParam);
										RegCloseKey(subKey);
										if (!lParam)
										{
											RegCloseKey(hk);
											extension = 0;
											goto again;
										}
									}
								}

								RegCloseKey(hk);

								// Convert the GUID string to a GUID and put it in MYARGS
								if (!lParam && (lParam = CLSIDFromString(&language[0], &MyArgs.Guid)))
									display_COM_error("Can't get engine GUID: %08X", lParam);
							}
						}

						// Now that we've got the GUID, and the script filename, run the script thread
						if (!lParam)
						{
							// NOTE: We really should have a critical section around this access to
							// ThreadHandle since the script thread also accesses it. It is left up to
							// you to implement this

							// Create a secondary thread to run the script. It will do the
							// actual call to the engine to run the script while we go back and
							// manage the user interface
							MyArgs.ThreadHandle = CreateThread(0, 0, runScript, &MyArgs, 0, &wParam);

							// NOTE: If script thread starts, 'ThreadHandle' will be closed by that
							// thread when it terminates
						}
					}

					break;	// End of IDM_FILE_RUNSCRIPT
				}
			}

			break;
		}

		// ******************************************************************
		case WM_APP:
		{
			// Our script thread posts a WM_APP if it needs us to display an error message.
			// wParam = A pointer to the string to display. If 0, then lParam is an error
			// number to be passed to display_sys_error().
			// lParam = The HRESULT. If 0, then wParam is an allocated WCHAR string which
			// we must free with GlobalFree()
			if (!wParam)
				display_sys_error((DWORD)lParam);
			else if (!lParam)
			{
				MessageBoxW(hwnd, (const WCHAR *)wParam, &ErrorStrWide[0], MB_OK|MB_ICONEXCLAMATION);
				GlobalFree((void *)wParam);
			}
			else
				display_COM_error((LPCTSTR)wParam, (HRESULT)lParam);

			return(0);
		}

		// ******************************************************************
		case WM_SETFOCUS:
		{
			SetFocus(GetDlgItem(hwnd, IDC_TRACE));
			break;
		}

		// ******************************************************************
		case WM_SIZE:
		{
			MoveWindow(GetDlgItem(hwnd, IDC_TRACE), 0, 0, LOWORD(lParam), HIWORD(lParam), TRUE);
			break;
		}

		// ******************************************************************
		// ====================== Create main window ========================
		case WM_INITDIALOG:
		{
			HICON	icon;

			// Load/set icon for System menu on the window. I put my icon
			// in this executable's resources. Note that windows frees this
			// when my window is closed
			if ((icon = LoadIcon(InstanceHandle,MAKEINTRESOURCE(IDI_MAIN_ICON))))
				SetClassLong(hwnd, GCL_HICON, (LONG)icon);
  
			SendDlgItemMessage(hwnd, IDC_TRACE, WM_SETFONT, (WPARAM)GetStockObject(DEFAULT_GUI_FONT), 0);

			return(1);
		}

		// ******************************************************************
		// =================== User wants to close window ===================
		case WM_CLOSE:
		{
			// NOTE: We really should have a critical section around this access to
			// ThreadHandle since the script thread also accesses it. It is left up to
			// you to implement this

			// Is a script running?
			if (MyArgs.ThreadHandle)
			{
				// Abort the script by calling InterruptScriptThread. This
				// is one of the few IActiveScript functions we can call by
				// any thread
				MyArgs.EngineActiveScript->lpVtbl->InterruptScriptThread(MyArgs.EngineActiveScript, SCRIPTTHREADID_ALL, 0, 0);

				// NOTE: Just because InterruptScriptThread has returned doesn't mean
				// that the thread has terminated. It simply means that the engine has
				// marked the running script for termination. We still have to "wait" for the
				// thread to terminate. We'll do that by testing when ThreadHandle is
				// 0. (Remember that the script thread zeroes it upon termination).
				// There's one other problem. If the script thread is somehow "sleeping"
				// or waiting for something itself, for example in a call to MessageBox,
				// then the engine will never get a chance to terminate it. To get
				// around this problem, we'll increment a count, and Sleep() in between
				// increments. When the count "times out", then we'll assume the script
				// is locked up, and brute force terminate it ourselves
				wParam = 0;
				while (MyArgs.ThreadHandle && ++wParam < 25) Sleep(100);
				if (MyArgs.ThreadHandle) TerminateThread(MyArgs.ThreadHandle, 0);
			}

			// Close this window
			DestroyWindow(hwnd);

			return(1);
		}

		case WM_DESTROY:
		{
 			// Post the WM_QUIT message to quit the message loop in WinMain()
			PostQuitMessage(0);

			return(1);
		}
	}

	// Indicate that I didn't handle the msg
	return(0);
} 





/************************** WinMain() **************************
 * Our EXE's entry point. This is called by Windows when our
 * EXE first starts up.
 */

int APIENTRY WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
	HRESULT				hr;

	// Save Instance handle which I need when opening the "Choose engine" dialog
	// box. That dialog is linked into this executable's resources, so I need to pass
	// this handle to DialogBox() when I open that dialog
	InstanceHandle = hInstance;

	// Loads Windows common control's DLL
	InitCommonControls();

	// MainWindow not open yet
	MainWindow = 0;

	// Initialize COM DLL
	if ((hr = CoInitializeEx(0, COINIT_MULTITHREADED)))
		display_COM_error("Failed to initialize COM: %08X", hr);
	else
	{
		// Initialize MyRealIActiveScriptSite object
		initIActiveScriptSiteObject();

		// No script thread running yet
		MyArgs.ThreadHandle = 0;

		// Create Main window
		if (!(MainWindow = CreateDialog(InstanceHandle, MAKEINTRESOURCE(IDD_MAINWINDOW), 0, mainWndProc)))
			display_sys_error(0);
		else
		{
			MSG		msg;

			// Show the window with default size/position
			ShowWindow(MainWindow, SW_SHOWDEFAULT);
			UpdateWindow(MainWindow);

			// Here's our message loop which is executed until the user closes
			// down our MainWindow
			while (GetMessage(&msg, 0, 0, 0) == 1)
			{
				TranslateMessage(&msg);
				DispatchMessage(&msg);
			}
		}

		// Allow our IActiveScriptSite to free any resource
		MyActiveScriptSite.site.lpVtbl->Release((IActiveScriptSite *)&MyActiveScriptSite);
	}

	// Free COM
	CoUninitialize();

	return(0);
}
