// This is the implementation of our proprietary IApp (multiple interface)
// object.
//
// An IApp object starts with the VTable our IDL file refers to
// as an IAppVTable interface. This VTable begins with the IUnknown and
// IDispatch functions. Then after the IDispatch functions are our own,
// proprietary functions that a script can call. The script engine gets an
// IApp object by passing the string "application" to our IActiveScriptSite's
// GetItemInfo function. That happens when the script references an object
// named "application".
//
// It contains the following functions that a script may call:
// Output() - Writes a line of text to our main window.
// SetTitle() - Sets the title of our main window.
// GetTitle() - Gets the title of our main window.

#define INITGUID
#include <windows.h>
#include <stddef.h>
#include <tchar.h>
#include <objbase.h>
#include <activscp.h>
#include <olectl.h>
#include <cguid.h>
#include "IActiveScriptSite.h"
#include "AppObject.h"
#include "Guids.h"
#include "Extern.h" 
#include "Resource.h"

#undef  INTERFACE
#define INTERFACE IApp
DECLARE_INTERFACE_ (INTERFACE, IDispatch)
{
	// IUnknown functions
	STDMETHOD  (QueryInterface)		(THIS_ REFIID, void **) PURE;
	STDMETHOD_ (ULONG, AddRef)		(THIS) PURE;
	STDMETHOD_ (ULONG, Release)		(THIS) PURE;
	// IDispatch functions
	STDMETHOD_ (ULONG, GetTypeInfoCount)(THIS_ UINT *) PURE;
	STDMETHOD_ (ULONG, GetTypeInfo)		(THIS_ UINT, LCID, ITypeInfo **) PURE;
	STDMETHOD_ (ULONG, GetIDsOfNames)	(THIS_ REFIID, LPOLESTR *, UINT, LCID, DISPID *) PURE;
	STDMETHOD_ (ULONG, Invoke)			(THIS_ DISPID, REFIID, LCID, WORD, DISPPARAMS *, VARIANT *, EXCEPINFO *, UINT *) PURE;
	// Extra functions
	STDMETHOD  (CreateDocument)	(THIS_ BSTR, IDispatch **) PURE;
};

// Since we need to add sub-objects, and other private data members
// to our app object, we'll define a MyRealIApp containing those
// sub-objects and private members. Our MyRealIApp contains the
// IApp (as the base object) and IProvideMultipleClassInfo
// sub-objects.
typedef struct {
	IApp						iApp;			// NOTE: Our IApp must be the base object.
	IProvideMultipleClassInfo	classInfo;		// Our IProvideMultipleClassInfo sub-object
} MyRealIApp;

// Our IProvideMultipleClassInfo VTable
static STDMETHODIMP QueryInterface_CInfo(IProvideMultipleClassInfo *, REFIID, void **);
static STDMETHODIMP_(ULONG) AddRef_CInfo(IProvideMultipleClassInfo *);
static STDMETHODIMP_(ULONG) Release_CInfo(IProvideMultipleClassInfo *);
static STDMETHODIMP GetClassInfo_CInfo(IProvideMultipleClassInfo *, ITypeInfo **);
static STDMETHODIMP GetGUID_CInfo(IProvideMultipleClassInfo *, DWORD, GUID *);
static STDMETHODIMP GetMultiTypeInfoCount_CInfo(IProvideMultipleClassInfo *, ULONG *);
static STDMETHODIMP GetInfoOfIndex_CInfo(IProvideMultipleClassInfo *, ULONG, DWORD, ITypeInfo **, DWORD *, ULONG *, GUID *, GUID *);

static const IProvideMultipleClassInfoVtbl IProvideMultipleClassInfoTable = {
	QueryInterface_CInfo,
	AddRef_CInfo,
	Release_CInfo,
	GetClassInfo_CInfo,
	GetGUID_CInfo,
	GetMultiTypeInfoCount_CInfo,
	GetInfoOfIndex_CInfo};

// Our IApp VTable. Starts with an IDispatch, and then contains proprietary
// functions that a script can call via IDispatch->Invoke()
static STDMETHODIMP QueryInterface(MyRealIApp *, REFIID, void **);
static STDMETHODIMP_(ULONG) AddRef(MyRealIApp *);
static STDMETHODIMP_(ULONG) Release(MyRealIApp *);
static STDMETHODIMP GetTypeInfoCount(MyRealIApp *, UINT *);
static STDMETHODIMP GetTypeInfo(MyRealIApp *, UINT , LCID , ITypeInfo **);
static STDMETHODIMP GetIDsOfNames(MyRealIApp *, REFIID, OLECHAR **, UINT, LCID, DISPID *);
static STDMETHODIMP Invoke(MyRealIApp *, DISPID, REFIID, LCID, WORD, DISPPARAMS *, VARIANT *, EXCEPINFO *, UINT *);
static STDMETHODIMP CreateDocument(MyRealIApp *, BSTR, IDispatch **);

static const IAppVtbl IAppTable = {
	QueryInterface,
	AddRef,
	Release,
	GetTypeInfoCount,
	GetTypeInfo,
	GetIDsOfNames,
	Invoke,
	CreateDocument};

// Our IDocument VTable. Starts with an IDispatch, and then contains proprietary
// functions that a script can call via IDispatch->Invoke()
static STDMETHODIMP _QueryInterface(void *, REFIID, void **);
static STDMETHODIMP_(ULONG) _AddRef(void *);
static STDMETHODIMP_(ULONG) Doc_Release(void *);
static STDMETHODIMP _GetTypeInfoCount(void *, UINT *);
static STDMETHODIMP _GetTypeInfo(void *, UINT , LCID , ITypeInfo **);
static STDMETHODIMP _GetIDsOfNames(void *, REFIID, OLECHAR **, UINT, LCID, DISPID *);
static STDMETHODIMP _Invoke(void *, DISPID, REFIID, LCID, WORD, DISPPARAMS *, VARIANT *, EXCEPINFO *, UINT *);
static STDMETHODIMP WriteText(MyRealIDocument *, BSTR);

static const IDocumentVtbl IDocumentTable = {
	_QueryInterface,
	_AddRef,
	Doc_Release,
	_GetTypeInfoCount,
	_GetTypeInfo,
	_GetIDsOfNames,
	_Invoke,
	WriteText};

// For our purposes, we need only one IApp object, so
// we'll declare it global
static MyRealIApp				MyIApp;

// The ITypeInfo for our IApp object. We need only 1 so we make it global
static ITypeInfo				*IAppObjectTypeInfo;

// The ITypeInfo for our IApp VTable. We need only one of these
static ITypeInfo				*IAppVTableTypeInfo;

// The ITypeInfo for our IDocument VTable. We need only one of these
static ITypeInfo				*IDocVTableTypeInfo;





// =======================================================================
// ========================== Helper functions ===========================
// =======================================================================

/******************* getAppObjectITypeInfo() ******************
 * Loads/creates the ITypeInfo for our Application object (IApp).
 *
 * RETURNS: Pointer to new MyRealIActiveScriptSite if success,
 * or 0 if fail.
 */

HRESULT getAppObjectITypeInfo(ITypeInfo **typeInfo)
{
	register HRESULT	hr;

	hr = S_OK;

	// Make sure we have an ITypeInfo for our IApp object
	if (!IAppObjectTypeInfo)
	{
		// Load the ITypeInfo for our "coClass IApp" (ie, the ITypeInfo for our
		// IApp object itself -- not one of the VTable in it). We pass IApp's
		// object GUID (CLSID_IApp)
		if ((hr = getITypeInfoFromExe(&CLSID_IApp, &IAppObjectTypeInfo))) goto bad;

		// AddRef() it so that it stays in memory until our IActiveScriptSite's Release()
		IAppObjectTypeInfo->lpVtbl->AddRef(IAppObjectTypeInfo);
	}

	// Return this ITypeInfo *
	*typeInfo = IAppObjectTypeInfo;
bad:
	return(hr);
}





/************************ getAppObject() **********************
 * Retrieves our IApp (actually, MyRealIApp) object.
 */

IUnknown * getAppObject(void)
{
	return((IUnknown *)&MyIApp);
}





/******************** initMyRealIAppObject() *******************
 * Initializes our global IApp (actually, MyRealIApp) object.
 * Called when our program starts up.
 */

void initMyRealIAppObject(void)
{
	// Initialize the sub-object VTable pointers
	MyIApp.iApp.lpVtbl = (IAppVtbl *)&IAppTable;
	MyIApp.classInfo.lpVtbl = (IProvideMultipleClassInfoVtbl *)&IProvideMultipleClassInfoTable;

	// Haven't yet loaded the ITypeInfo's for IApp's VTable, nor IApp itself
	IDocVTableTypeInfo = IAppObjectTypeInfo = IAppVTableTypeInfo = 0;
}





/******************** freeMyRealIAppObject() *******************
 * Frees our IApp resources. Called when our program exits.
 */

void freeMyRealIAppObject(void)
{
	// Free the ITypeInfos if loaded
	if (IDocVTableTypeInfo) IDocVTableTypeInfo->lpVtbl->Release(IDocVTableTypeInfo);
	if (IAppVTableTypeInfo) IAppVTableTypeInfo->lpVtbl->Release(IAppVTableTypeInfo);
	if (IAppObjectTypeInfo) IAppObjectTypeInfo->lpVtbl->Release(IAppObjectTypeInfo);
}





// =======================================================================
// =========================== IApp functions ============================
// =======================================================================

static STDMETHODIMP QueryInterface(MyRealIApp *this, REFIID vTableGuid, void **ppv) 
{
	// Does he want our IApp IDispatch or IUnknown?
	if (IsEqualIID(vTableGuid, &IID_IUnknown) || IsEqualIID(vTableGuid, &IID_IDispatch))
		*ppv = this;

	// Does he want our IApp IProvideMultipleClassInfo, IProvideClassInfo2, or IProvideClassInfo?
	else if (IsEqualIID(vTableGuid, &IID_IProvideMultipleClassInfo) || IsEqualIID(vTableGuid, &IID_IProvideClassInfo2) || IsEqualIID(vTableGuid, &IID_IProvideClassInfo))
		*ppv = ((char *)this + offsetof(MyRealIApp, classInfo));

	// Our IApp doesn't implement any other objects
	else
	{
		*ppv = 0;
		return(E_NOINTERFACE);
	}
 
	return(S_OK);
}

static STDMETHODIMP_(ULONG) AddRef(MyRealIApp *this)
{
	return(1);
}
 
static STDMETHODIMP_(ULONG) Release(MyRealIApp *this)
{
	return(1);
}

static STDMETHODIMP GetTypeInfoCount(MyRealIApp *this, UINT *pctinfo)
{
	*pctinfo = 1;
	return(S_OK);
}
 
static STDMETHODIMP GetTypeInfo(MyRealIApp *this, UINT ctinfo, LCID lcid, ITypeInfo **typeInfo)
{
	register HRESULT	hr;

	// Make sure our IApp VTable's ITypeInfo is loaded, so the engine
	// can get information about the extra functions we added to the end
	// of the IApp VTable
	if (!IAppVTableTypeInfo && (hr = getITypeInfoFromExe(&IID_IApp, &IAppVTableTypeInfo))) return(hr);

	*typeInfo = IAppVTableTypeInfo;

	IAppVTableTypeInfo->lpVtbl->AddRef(IAppVTableTypeInfo);

	return(S_OK);
}

static STDMETHODIMP GetIDsOfNames(MyRealIApp *this, REFIID riid, OLECHAR **rgszNames, UINT cNames, LCID lcid, DISPID *rgdispid)
{
	register HRESULT	hr;

	// Make sure we have the ITypeInfo for our IApp
	if (!IAppVTableTypeInfo && (hr = getITypeInfoFromExe(&IID_IApp, &IAppVTableTypeInfo))) return(hr);

	// Return the DISPID of the function the caller wants. Because we have made this a dual
	// VTable, we can use our ITypeInfo to do all the work of returning the DISPID
	return(IAppVTableTypeInfo->lpVtbl->GetIDsOfNames(IAppVTableTypeInfo, rgszNames, cNames, rgdispid));
}
 
static STDMETHODIMP Invoke(MyRealIApp *this, DISPID id, REFIID riid, LCID lcid, WORD flag, DISPPARAMS *params, VARIANT *ret, EXCEPINFO *pei, UINT *pu)
{
	register HRESULT	hr;

	if (!IAppVTableTypeInfo && (hr = getITypeInfoFromExe(&IID_IApp, &IAppVTableTypeInfo))) return(hr);

	// Call one of the extra functions we added to IApp's VTable. Because we have made
	// this a dual VTable, we can use our ITypeInfo to do all the work of calling the function
	return(IAppVTableTypeInfo->lpVtbl->Invoke(IAppVTableTypeInfo, this, id, flag, params, ret, pei, pu));
}

// =================== Extra IApp functions (added to end of VTable) ==============
// These are the extra functions added to the end of IApp's VTable (and
// which are called by the script engine through our Invoke function above).

static STDMETHODIMP CreateDocument(MyRealIApp *this, BSTR bstr, IDispatch **obj)
{
	// Assume an error
	*obj = 0;

	// NOTE: We use SendMessage, so this call will not return until
	// MainWindow's message handler processes it. This is ok because
	// we know that our MainWindow handling of this message does not
	// "wait" or "sleep", plus if MainWindow has to abort the script,
	// it will empty out the queue of any WM_APP+2 messages
	SendMessage(MainWindow, WM_APP+2, (WPARAM)obj, (LPARAM)bstr);

	if (*obj) return(S_OK);
	return(E_FAIL);
}
















// =======================================================================
// ================== IApp's IProvideMultipleClassInfo ==================
// =======================================================================

static STDMETHODIMP QueryInterface_CInfo(IProvideMultipleClassInfo *this, REFIID vTableGuid, void **ppv)
{
	// Because our IProvideMultipleClassInfo is a sub-object of our IApp, we
	// need to delegate to our IApp's QueryInterface. Since our
	// IProvideMultipleClassInfo sub-object is embedded right inside our
	// MyRealIApp, we can use pointer arithematic to get our IApp
	return(QueryInterface((MyRealIApp *)((char *)this - offsetof(MyRealIApp, classInfo)), vTableGuid, ppv));
}

static STDMETHODIMP_(ULONG) AddRef_CInfo(IProvideMultipleClassInfo *this)
{
	return(AddRef((MyRealIApp *)((char *)this - offsetof(MyRealIApp, classInfo))));
}

static STDMETHODIMP_(ULONG) Release_CInfo(IProvideMultipleClassInfo *this)
{
	return(Release((MyRealIApp *)((char *)this - offsetof(MyRealIApp, classInfo))));
}

static STDMETHODIMP GetClassInfo_CInfo(IProvideMultipleClassInfo *this, ITypeInfo **classITypeInfo)
{
	register HRESULT	hr;

	// Make sure we have an ITypeInfo for the coClass object for this IDispatch (ie, IApp's ITypeInfo.
	// not the ITypeInfo of one of its VTables)
	if (!(hr = getAppObjectITypeInfo(classITypeInfo)))
	{
		// AddRef() it on behalf of the caller
		(*classITypeInfo)->lpVtbl->AddRef(*classITypeInfo);
	}

	return(hr);
}

static STDMETHODIMP GetGUID_CInfo(IProvideMultipleClassInfo *this, DWORD guidType, GUID *guid)
{
	// Caller wants the [default, source] VTable GUID of this IDispatch's object?
	// Our app object has no such VTable in it
	if (guidType == GUIDKIND_DEFAULT_SOURCE_DISP_IID)
		return(E_NOTIMPL);
	return(E_INVALIDARG);
}

static STDMETHODIMP GetMultiTypeInfoCount_CInfo(IProvideMultipleClassInfo *this, ULONG *count)
{
	// This IDispatch is for only one coClass object (IApp)
	*count = 1;
	return(S_OK);
}

static STDMETHODIMP GetInfoOfIndex_CInfo(IProvideMultipleClassInfo *this, ULONG objNum, DWORD flags,
								   ITypeInfo **classITypeInfo, DWORD *retFlags, ULONG *reservedIds,
								   GUID *defVTableGuid, GUID *defSrcVTableGuid)
{
	register HRESULT	hr;

	hr = S_OK;

	*retFlags = 0;

	// Because we have only 1 coClass object for this IDispatch, we
	// can ignore the objNum arg. It should always be 0 (for the first
	// coClass object for this IDispatch). If IDispatch was for more than
	// one coClass object, then we'd need to number each coClass object,
	// and return info for whichever coClass object corresponded to
	// the "objNum" specified by the caller.

	// Does caller want us to tell him how many DISPIDs we have
	// reserved for the extra functions wrapped by our IDispatch?
	if (flags & MULTICLASSINFO_GETNUMRESERVEDDISPIDS)
	{
		*reservedIds = 2;		// Set this to the highest DISPID our [default] VTable uses
		*retFlags = MULTICLASSINFO_GETNUMRESERVEDDISPIDS;
	}

	// Does caller want us to return the GUID for the [default]
	// VTable for the object that this IDispatch belongs to?
	if (flags & MULTICLASSINFO_GETIIDPRIMARY)
	{
		CopyMemory(defVTableGuid, &IID_IApp, sizeof(GUID));
		*retFlags |= MULTICLASSINFO_GETIIDPRIMARY;
	}

	// Does caller want us to return the GUID for the [default, source]
	// VTable for the object that this IDispatch belongs to?
//	if (flags & MULTICLASSINFO_GETIIDSOURCE)
//	{
//		// Here, we'd copy the GUID of our [default, source] VTable to defSrcVTableGuid
//		*retFlags |= MULTICLASSINFO_GETIIDSOURCE;
//	}

	// Does caller want us to give him a pointer to the ITypeInfo for the
	// class object to which this IDispatch belongs?
	if (flags & MULTICLASSINFO_GETTYPEINFO)
	{
		// Make sure we have an ITypeInfo for our IApp object
		if ((hr = getAppObjectITypeInfo(classITypeInfo))) goto bad;

		// AddRef() it on behalf of the caller
		(*classITypeInfo)->lpVtbl->AddRef(*classITypeInfo);

		*retFlags |= MULTICLASSINFO_GETTYPEINFO;
	}

bad:
	return(hr);
}

















/************************ allocIDocument() **********************
 * Allocates our IDocument (actually, MyRealIDocument) object.
 *
 * obj = Where to return the IDocument pointer.
 */

HRESULT allocIDocument(IDispatch **obj)
{
	register MyRealIDocument	*doc;
	register HRESULT			hr;

	hr = S_OK;

	// Make sure our lIDocument VTable's ITypeInfo is loaded, so the engine
	// can get information about the extra functions we added to the end
	// of the IDocument VTable
	if (IDocVTableTypeInfo || !(hr = getITypeInfoFromExe(&IID_IDocument, &IDocVTableTypeInfo)))
	{
		// Allocate a MyRealIDocument
		if (!(doc = GlobalAlloc(GMEM_FIXED, sizeof(MyRealIDocument))))
			hr = E_OUTOFMEMORY;
		else
		{
			// Set the VTable
			doc->iDocument.lpVtbl = (IDocumentVtbl *)&IDocumentTable;

			// Store a pointer to the ITypeInfo
			doc->TypeInfo = IDocVTableTypeInfo;

			// Window not yet open
			doc->Hwnd = 0;

			// Text not yet written
			doc->Text = 0;

			// Increment RefCount -- once because this document window's WM_DESTROY
			// will call the IDocument->Release(), and also increment RefCount on
			// behalf of the script (ie, we'll be returning the IDocument to the
			// script)
			doc->RefCount = 2;

			*obj = (IDispatch *)doc;
		}
	}

	return(hr);
}




// =======================================================================
// ========================= IDocument functions =========================
// =======================================================================

static STDMETHODIMP _QueryInterface(void *this, REFIID vTableGuid, void **ppv) 
{
	// Does he want our IDocument IDispatch or IUnknown?
	if (IsEqualIID(vTableGuid, &IID_IUnknown) || IsEqualIID(vTableGuid, &IID_IDispatch))
		*ppv = this;

	// Our IDocument doesn't implement any other objects
	else
	{
		*ppv = 0;
		return(E_NOINTERFACE);
	}
 
	return(S_OK);
}

static STDMETHODIMP_(ULONG) _AddRef(void *this)
{
	return(++((MyRealIDocument *)this)->RefCount);
}
 
static STDMETHODIMP_(ULONG) Doc_Release(void *this)
{
	register MyRealIDocument	*doc;

	doc = (MyRealIDocument *)this;

	if (--((MyRealIDocument *)this)->RefCount == 0)
	{
		if (doc->Text) GlobalFree(doc->Text);
		GlobalFree(this);
		return(0);
	}

	return(doc->RefCount);
}

static STDMETHODIMP _GetTypeInfoCount(void *this, UINT *pctinfo)
{
	*pctinfo = 1;
	return(S_OK);
}
 
static STDMETHODIMP _GetTypeInfo(void *this, UINT ctinfo, LCID lcid, ITypeInfo **typeInfo)
{
	*typeInfo = ((MyRealIDocument *)this)->TypeInfo;
	(*typeInfo)->lpVtbl->AddRef(*typeInfo);
	return(S_OK);
}

static STDMETHODIMP _GetIDsOfNames(void *this, REFIID riid, OLECHAR **rgszNames, UINT cNames, LCID lcid, DISPID *rgdispid)
{
	// Return the DISPID of the function the caller wants. Because we have made this a dual
	// VTable, we can use our ITypeInfo to do all the work of returning the DISPID
	return(((MyRealIDocument *)this)->TypeInfo->lpVtbl->GetIDsOfNames(((MyRealIDocument *)this)->TypeInfo, rgszNames, cNames, rgdispid));
}
 
static STDMETHODIMP _Invoke(void *this, DISPID id, REFIID riid, LCID lcid, WORD flag, DISPPARAMS *params, VARIANT *ret, EXCEPINFO *pei, UINT *pu)
{
	// Call one of the extra functions we added to IApp's VTable. Because we have made
	// this a dual VTable, we can use our ITypeInfo to do all the work of calling the function
	return(((MyRealIDocument *)this)->TypeInfo->lpVtbl->Invoke(((MyRealIDocument *)this)->TypeInfo, this, id, flag, params, ret, pei, pu));
}

// =================== Extra IDocument functions (added to end of VTable) ==============
// These are the extra functions added to the end of IDocument's VTable (and
// which are called by the script engine through our Invoke function above).

static STDMETHODIMP WriteText(MyRealIDocument *this, BSTR bstr)
{
	register DWORD		len;

	// Free any previous text
	if (this->Text) GlobalFree(this->Text);

	// Make a copy of the passed text
#ifdef UNICODE
	len = SysStringByteLen(bstr) + sizeof(WCHAR);
	if (!(this->Text = (LPTSTR)GlobalAlloc(GMEM_FIXED, len))) return(E_OUTOFMEMORY);
	lstrcpy(this->Text, bstr);
#else
	len = WideCharToMultiByte(CP_ACP, 0, bstr, -1, 0, 0, 0, 0) + 1;
	if (!(this->Text = (LPTSTR)GlobalAlloc(GMEM_FIXED, len))) return(E_OUTOFMEMORY);
	len = WideCharToMultiByte(CP_ACP, 0, bstr, -1, this->Text, len, 0, 0);
#endif

	// Redraw the window
	if (this->Hwnd) InvalidateRect(this->Hwnd, 0, 1);

	return(S_OK);
}
