// This is the implementation of our proprietary IApp (multiple interface)
// object.
//
// An IApp object starts with the VTable our IDL file refers to
// as an IAppVTable interface. This VTable begins with the IUnknown and
// IDispatch functions. Then after the IDispatch functions are our own,
// proprietary functions that a script can call. The script engine gets an
// IApp object by passing the string "application" to our IActiveScriptSite's
// GetItemInfo function. That happens when the script references an object
// named "application".
//
// It contains the following functions that a script may call:
// Output() - Writes a line of text to our main window.
// SetTitle() - Sets the title of our main window.
// GetTitle() - Gets the title of our main window.

#define INITGUID
#include <windows.h>
#include <stddef.h>
#include <tchar.h>
#include <objbase.h>
#include <activscp.h>
#include <olectl.h>
#include <cguid.h>
#include "IActiveScriptSite.h"
#include "AppObject.h"
#include "Guids.h"
#include "Extern.h" 
#include "Resource.h"

// For our purposes, we need only one IApp object, so
// we'll declare it global
static MyRealIApp				MyIApp;

// The ITypeInfo for our IApp object. We need only 1 so we make it global
static ITypeInfo				*IAppObjectTypeInfo;

// The ITypeInfo for our IApp VTable. We need only one of these
static ITypeInfo				*IAppVTableTypeInfo;

// The ITypeInfo for our IDocument VTable. We need only one of these
static ITypeInfo				*IDocVTableTypeInfo;

const wchar_t					MyRealIApp::MyAppObjectName[] =	L"application";





// =======================================================================
// ========================== Helper functions ===========================
// =======================================================================

/******************* getAppObjectITypeInfo() ******************
 * Loads/creates the ITypeInfo for our Application object (IApp).
 *
 * RETURNS: Pointer to new MyRealIActiveScriptSite if success,
 * or 0 if fail.
 */

HRESULT getAppObjectITypeInfo(ITypeInfo **typeInfo)
{
	register HRESULT	hr;

	hr = S_OK;

	// Make sure we have an ITypeInfo for our IApp object
	if (!IAppObjectTypeInfo)
	{
		// Load the ITypeInfo for our "coClass IApp" (ie, the ITypeInfo for our
		// IApp object itself -- not one of the VTable in it). We pass IApp's
		// object GUID (CLSID_IApp)
		if ((hr = getITypeInfoFromExe(CLSID_IApp, &IAppObjectTypeInfo))) goto bad;

		// AddRef() it so that it stays in memory until our IActiveScriptSite's Release()
		IAppObjectTypeInfo->AddRef();
	}

	// Return this ITypeInfo *
	*typeInfo = IAppObjectTypeInfo;
bad:
	return(hr);
}





/************************ getAppObject() **********************
 * Retrieves our IApp (actually, MyRealIApp) object.
 */

IUnknown * getAppObject(void)
{
	return(static_cast<IDispatch *>(&MyIApp));
}





/******************** initMyRealIAppObject() *******************
 * Initializes our global IApp (actually, MyRealIApp) object.
 * Called when our program starts up.
 */

void initMyRealIAppObject(void)
{
	// Haven't yet loaded the ITypeInfo's for IApp's VTable, nor IApp itself
	IDocVTableTypeInfo = IAppObjectTypeInfo = IAppVTableTypeInfo = 0;
}





/******************** freeMyRealIAppObject() *******************
 * Frees our IApp resources. Called when our program exits.
 */

void freeMyRealIAppObject(void)
{
	// Free the ITypeInfos if loaded
	if (IDocVTableTypeInfo) IDocVTableTypeInfo->Release();
	if (IAppVTableTypeInfo) IAppVTableTypeInfo->Release();
	if (IAppObjectTypeInfo) IAppObjectTypeInfo->Release();
}





// =======================================================================
// =========================== IApp functions ============================
// =======================================================================

STDMETHODIMP MyRealIApp::QueryInterface(REFIID vTableGuid, void **ppv) 
{
	// Does he want our IApp IDispatch or IUnknown?
	if (IsEqualIID(vTableGuid, IID_IUnknown) || IsEqualIID(vTableGuid, IID_IDispatch))
		*ppv = static_cast<IDispatch *>(this);

	// Does he want our IApp IProvideMultipleClassInfo, IProvideClassInfo2, or IProvideClassInfo?
	else if (IsEqualIID(vTableGuid, IID_IProvideMultipleClassInfo) || IsEqualIID(vTableGuid, IID_IProvideClassInfo2) || IsEqualIID(vTableGuid, IID_IProvideClassInfo))
		*ppv = static_cast<IProvideMultipleClassInfo *>(this);

	// Our IApp doesn't implement any other objects
	else
	{
		*ppv = 0;
		return(E_NOINTERFACE);
	}
 
	return(S_OK);
}

STDMETHODIMP_(ULONG) MyRealIApp::AddRef()
{
	return(1);
}
 
STDMETHODIMP_(ULONG) MyRealIApp::Release()
{
	return(1);
}

STDMETHODIMP MyRealIApp::GetTypeInfoCount(UINT *pctinfo)
{
	*pctinfo = 1;
	return(S_OK);
}
 
STDMETHODIMP MyRealIApp::GetTypeInfo(UINT ctinfo, LCID lcid, ITypeInfo **typeInfo)
{
	register HRESULT	hr;

	// Make sure our IApp VTable's ITypeInfo is loaded, so the engine
	// can get information about the extra functions we added to the end
	// of the IApp VTable
	if (!IAppVTableTypeInfo && (hr = getITypeInfoFromExe(IID_IApp, &IAppVTableTypeInfo))) return(hr);

	*typeInfo = IAppVTableTypeInfo;

	IAppVTableTypeInfo->AddRef();

	return(S_OK);
}

STDMETHODIMP MyRealIApp::GetIDsOfNames(REFIID riid, OLECHAR **rgszNames, UINT cNames, LCID lcid, DISPID *rgdispid)
{
	register HRESULT	hr;

	// Make sure we have the ITypeInfo for our IApp
	if (!IAppVTableTypeInfo && (hr = getITypeInfoFromExe(IID_IApp, &IAppVTableTypeInfo))) return(hr);

	// Return the DISPID of the function the caller wants. Because we have made this a dual
	// VTable, we can use our ITypeInfo to do all the work of returning the DISPID
	return(IAppVTableTypeInfo->GetIDsOfNames(rgszNames, cNames, rgdispid));
}
 
STDMETHODIMP MyRealIApp::Invoke(DISPID id, REFIID riid, LCID lcid, WORD flag, DISPPARAMS *params, VARIANT *ret, EXCEPINFO *pei, UINT *pu)
{
	register HRESULT	hr;

	if (!IAppVTableTypeInfo && (hr = getITypeInfoFromExe(IID_IApp, &IAppVTableTypeInfo))) return(hr);

	// Call one of the extra functions we added to IApp's VTable. Because we have made
	// this a dual VTable, we can use our ITypeInfo to do all the work of calling the function
	return(IAppVTableTypeInfo->Invoke(this, id, flag, params, ret, pei, pu));
}

// =================== Extra IApp functions (added to end of VTable) ==============
// These are the extra functions added to the end of IApp's VTable (and
// which are called by the script engine through our Invoke function above).

STDMETHODIMP MyRealIApp::CreateDocument(BSTR bstr, IDispatch **obj)
{
	// Assume an error
	*obj = 0;

	// NOTE: We use SendMessage, so this call will not return until
	// MainWindow's message handler processes it. This is ok because
	// we know that our MainWindow handling of this message does not
	// "wait" or "sleep", plus if MainWindow has to abort the script,
	// it will empty out the queue of any WM_APP+2 messages
	::SendMessage(MainWindow, WM_APP+2, (WPARAM)obj, (LPARAM)bstr);

	if (*obj) return(S_OK);
	return(E_FAIL);
}
















// =======================================================================
// ================== IApp's IProvideMultipleClassInfo ==================
// =======================================================================

STDMETHODIMP MyRealIApp::GetClassInfo(ITypeInfo **classITypeInfo)
{
	register HRESULT	hr;

	// Make sure we have an ITypeInfo for the coClass object for this IDispatch (ie, IApp's ITypeInfo.
	// not the ITypeInfo of one of its VTables)
	if (!(hr = getAppObjectITypeInfo(classITypeInfo)))
	{
		// AddRef() it on behalf of the caller
		(*classITypeInfo)->AddRef();
	}

	return(hr);
}

STDMETHODIMP MyRealIApp::GetGUID(DWORD guidType, GUID *guid)
{
	// Caller wants the [default, source] VTable GUID of this IDispatch's object?
	// Our app object has no such VTable in it
	if (guidType == GUIDKIND_DEFAULT_SOURCE_DISP_IID)
		return(E_NOTIMPL);
	return(E_INVALIDARG);
}

STDMETHODIMP MyRealIApp::GetMultiTypeInfoCount(ULONG *count)
{
	// This IDispatch is for only one coClass object (IApp)
	*count = 1;
	return(S_OK);
}

STDMETHODIMP MyRealIApp::GetInfoOfIndex(ULONG objNum, DWORD flags,
								   ITypeInfo **classITypeInfo, DWORD *retFlags, ULONG *reservedIds,
								   GUID *defVTableGuid, GUID *defSrcVTableGuid)
{
	register HRESULT	hr;

	hr = S_OK;

	*retFlags = 0;

	// Because we have only 1 coClass object for this IDispatch, we
	// can ignore the objNum arg. It should always be 0 (for the first
	// coClass object for this IDispatch). If IDispatch was for more than
	// one coClass object, then we'd need to number each coClass object,
	// and return info for whichever coClass object corresponded to
	// the "objNum" specified by the caller.

	// Does caller want us to tell him how many DISPIDs we have
	// reserved for the extra functions wrapped by our IDispatch?
	if (flags & MULTICLASSINFO_GETNUMRESERVEDDISPIDS)
	{
		*reservedIds = 2;		// Set this to the highest DISPID our [default] VTable uses
		*retFlags = MULTICLASSINFO_GETNUMRESERVEDDISPIDS;
	}

	// Does caller want us to return the GUID for the [default]
	// VTable for the object that this IDispatch belongs to?
	if (flags & MULTICLASSINFO_GETIIDPRIMARY)
	{
		CopyMemory(defVTableGuid, &IID_IApp, sizeof(GUID));
		*retFlags |= MULTICLASSINFO_GETIIDPRIMARY;
	}

	// Does caller want us to return the GUID for the [default, source]
	// VTable for the object that this IDispatch belongs to?
//	if (flags & MULTICLASSINFO_GETIIDSOURCE)
//	{
//		// Here, we'd copy the GUID of our [default, source] VTable to defSrcVTableGuid
//		*retFlags |= MULTICLASSINFO_GETIIDSOURCE;
//	}

	// Does caller want us to give him a pointer to the ITypeInfo for the
	// class object to which this IDispatch belongs?
	if (flags & MULTICLASSINFO_GETTYPEINFO)
	{
		// Make sure we have an ITypeInfo for our IApp object
		if ((hr = getAppObjectITypeInfo(classITypeInfo))) goto bad;

		// AddRef() it on behalf of the caller
		(*classITypeInfo)->AddRef();

		*retFlags |= MULTICLASSINFO_GETTYPEINFO;
	}

bad:
	return(hr);
}

















/************************ allocIDocument() **********************
 * Allocates our IDocument (actually, MyRealIDocument) object.
 *
 * obj = Where to return the IDocument pointer.
 */

HRESULT allocIDocument(MyRealIDocument **obj)
{
	register MyRealIDocument	*doc;
	register HRESULT			hr;

	hr = S_OK;

	// Make sure our lIDocument VTable's ITypeInfo is loaded, so the engine
	// can get information about the extra functions we added to the end
	// of the IDocument VTable
	if (IDocVTableTypeInfo || !(hr = getITypeInfoFromExe(IID_IDocument, &IDocVTableTypeInfo)))
	{
		// Allocate a MyRealIDocument
		if (!(doc = new MyRealIDocument()))
			hr = E_OUTOFMEMORY;
		else
			*obj = doc;
	}

	return(hr);
}




// =======================================================================
// ========================= IDocument functions =========================
// =======================================================================

STDMETHODIMP MyRealIDocument::QueryInterface(REFIID vTableGuid, void **ppv) 
{
	// Does he want our IDocument IDispatch or IUnknown?
	if (IsEqualIID(vTableGuid, IID_IUnknown) || IsEqualIID(vTableGuid, IID_IDispatch))
		*ppv = static_cast<IDispatch *>(this);

	// Our IDocument doesn't implement any other objects
	else
	{
		*ppv = 0;
		return(E_NOINTERFACE);
	}
 
	return(S_OK);
}

STDMETHODIMP_(ULONG) MyRealIDocument::AddRef(void)
{
	return(++RefCount);
}
 
STDMETHODIMP_(ULONG) MyRealIDocument::Release(void)
{
	if (--RefCount == 0)
	{
		if (Text) GlobalFree(Text);
		delete this;
		return(0);
	}

	return(RefCount);
}

STDMETHODIMP MyRealIDocument::GetTypeInfoCount(UINT *pctinfo)
{
	*pctinfo = 1;
	return(S_OK);
}
 
STDMETHODIMP MyRealIDocument::GetTypeInfo(UINT ctinfo, LCID lcid, ITypeInfo **typeInfo)
{
	*typeInfo = IDocVTableTypeInfo;
	IDocVTableTypeInfo->AddRef();
	return(S_OK);
}

STDMETHODIMP MyRealIDocument::GetIDsOfNames(REFIID riid, OLECHAR **rgszNames, UINT cNames, LCID lcid, DISPID *rgdispid)
{
	// Return the DISPID of the function the caller wants. Because we have made this a dual
	// VTable, we can use our ITypeInfo to do all the work of returning the DISPID
	return(IDocVTableTypeInfo->GetIDsOfNames(rgszNames, cNames, rgdispid));
}
 
STDMETHODIMP MyRealIDocument::Invoke(DISPID id, REFIID riid, LCID lcid, WORD flag, DISPPARAMS *params, VARIANT *ret, EXCEPINFO *pei, UINT *pu)
{
	// Call one of the extra functions we added to IApp's VTable. Because we have made
	// this a dual VTable, we can use our ITypeInfo to do all the work of calling the function
	return(IDocVTableTypeInfo->Invoke(this, id, flag, params, ret, pei, pu));
}

// =================== Extra IDocument functions (added to end of VTable) ==============
// These are the extra functions added to the end of IDocument's VTable (and
// which are called by the script engine through our Invoke function above).

STDMETHODIMP MyRealIDocument::WriteText(BSTR bstr)
{
	register DWORD		len;

	// Free any previous text
	if (Text) GlobalFree(Text);

	// Make a copy of the passed text
#ifdef UNICODE
	len = ::SysStringByteLen(bstr) + sizeof(WCHAR);
	if (!(Text = (LPTSTR)::GlobalAlloc(GMEM_FIXED, len))) return(E_OUTOFMEMORY);
	lstrcpy(Text, bstr);
#else
	len = ::WideCharToMultiByte(CP_ACP, 0, bstr, -1, 0, 0, 0, 0) + 1;
	if (!(Text = (LPTSTR)::GlobalAlloc(GMEM_FIXED, len))) return(E_OUTOFMEMORY);
	len = ::WideCharToMultiByte(CP_ACP, 0, bstr, -1, Text, len, 0, 0);
#endif

	// Redraw the window
	if (Hwnd) ::InvalidateRect(Hwnd, 0, 1);

	return(S_OK);
}
