# -*- coding: cp1251 -*-
"""
    pygments.lexers.1S
    ~~~~~~~~~~~~~~~~~~~~~~~~

    Lexers for language: 1S.

    :license: GNU LGPL, see LICENSE for more details.
"""

import re

from pygments.lexer import RegexLexer, RegexLexerMeta, include, bygroups, using, this
from pygments.token import \
     Text, Comment, Operator, Keyword, Name, String, Number, Literal


__all__ = ['OneSLexer']


class OneSLexer(RegexLexer):

    name = '1S'
    aliases = ['1s']
    filenames = ['*.1s', '*.prm', '*.1cpp']
    mimetypes = ['text/x-1s']

    #: optional Comment or Whitespace
    _ws = r'(?:\s|//.*?\n|/[*].*?[*]/)+'


    flags = re.IGNORECASE | re.MULTILINE | re.DOTALL | re.UNICODE
    tokens = {
        'whitespace': [
            (r'^\s*#', Comment.Preproc, 'macro'),
            (r'^\s*//#.*?\n', Comment.Preproc),
            (r'\n', Text),
            (r'\s+', Text),
            (r'\\\n', Text), # line continuation
            (r'//.*?\n', Comment),
        ],
        'statements': [
            (r'L?"', String, 'string'),
            (r'(\s)*?\|', String, 'string'),
            (r'(0x[0-9a-fA-F]|0[0-7]+|(\d+\.\d*|\.\d+)|\d+)'
             r'e[+-]\d+[lL]?', Number.Float),
            (r'0x[0-9a-fA-F]+[Ll]?', Number.Hex),
            (r'0[0-7]+[Ll]?', Number.Oct),
            (r'(\d+\.\d*|\.\d+)', Number.Float),
            (r'\d+', Number.Integer),
            (r'\'[0-3][0-9]\.[0-1][0-9]\.([0-9][0-9]|[0-9][0-9][0-9][0-9])\'', Literal.Date),
            (r'[!%^&*()+=|\[\]:,.<>/?-]', Operator),
            (r'(||procedure|function)(\s+)', bygroups(Keyword, Text), 'funcname'),
            (r'(|Var||If||Then||Elsif||Else||Endlf|'
             r'|Do||For||To||While|'
             r'He|Not||Try||Except||'
             r'EndTry||Raise||Val||EndDo||Context|'
             r'|GetErrorDescription||Var||Goto||Return||'
             r'Continue||Break||And||Or||MetaData)\b', Keyword.Reserved),
            #(r'(class|)(\s+)', bygroups(Keyword, Text), 'class'), #  1++
            (r'(|GetBaseClass||' #   1++
            r'|SendMessageOwnMod||'
            r'GetPathName||GetEnvContext|'
            r'|GetParamsList||'
            r'SetOnIndexParams|||'
            r'ReplaceInstBaseClasses|_|_OnOpen|_|'
            r'_Throw|_|_GetCode|_SQLCreate)\b', Keyword.Reserved),
            (r'(|IndexedTable||ActiveX|' #   1++
            r'Win|WorkAsRegisterWin||'
            r'ExecuteModule||Delegate||EventManager|'
            r'|Struct|DynaValue||Thread|GUID|BinaryData'
            r'ODBCDataBase|ODBCRecordSet|MetaDataWork|SQLLock)\b', Name.Class),
            (r'(|PageBreak|LineBreak|'
            r'|TabSymbol)\b', Keyword.Constant),
            (r'(|Number||String||Date||Undefine|void|' #  
            r'|ValueTable||ValueList|'
            r'|Undefine||Query||Const|'
            r'|Reference||Enum| Document|' 
            r'|Register||ChartOfAccounts||Account|'
            r'|SubcontoKind||Operation||'
            r'BookkeepingTotals||CalcJournal||'
            r'CalculationKind||CalculationGroup||'
            r'Calendar||Query||Text||Table||'
            r'ValueList||ValueTable||Picture|'
            r'|iodic||FS|XBase|Xbase)\b', Keyword.Type),
            (r'(|Round||Int||Min||Max|10||Ln||StrLen' #  
            r'|IsBlankString||TrimL||TrimR||TrimAll|' #  
            r'|Left||Right||Mid||Find||StrReplace|'
            r'|StrCountOccur||StrLineCount|'
            r'|StrGetLine||Upper||Lower|OemToAnsi|'
            r'AnsiToOem||Chr||As|'
            r'|WorkingDate||CurDate||AddMonth|' #    
            r'|BegOfMonth||EndOfMonth||BegOfQuart||'
            r'EndOfQuart||BegOfYear||EndOfYear||BegOfWeek||'
            r'EndOfWeek||GetYear||GetMonbh||GetDay||'
            r'GetWeekOfYear||GetDayOfYear||GetDayOfWeek||'
            r'iodStr||BegOfStandrdRange||'
            r'|CurrentTime|' #    
            r'|MakeDocPosition||SplitDocPosition|' #     
            r'|Spelling||Format||Template||FixTemplate' #    
            r'|InputValue||InputNumeric||InputString|' #      
            r'|InputDate||Inputiod||InputEnum|'
            r'|DoQueryBox||DoMessageBox||Message|' #    
            r'|ClearMessageWindow||Status||||Dim|'
            r'|SystemCaption||ComputerName||' #   
            r'UserName||UserFullName||RightName|'
            r'|AccessRight||UserInterfaceName||'
            r'UserDir||IBDir||BinDir||'
            r'TempFilesDir||ExclusiveMode||GeneralLanguage|'
            r'|BeginTransaction||CommitTransation|' #    
            r'|RollBackTransaction|'
            r'|CreateObject||ReturnStatus||' #    
            r'OpenForm||OpenFormModal||ValueType||'
            r'ValueTypeStr||EmptyValue||GetEmptyValue|'
            r'|SetKind||LogMessageWrite||'
            r'AutoNumPrefix||GetSelectionValues||System|'
            r'|RunApp||ExitSystem||'
            r'FindMarkedForDelete||FindReferences||DeleteObjects|'
            r'|IdleProcessing|'
            r'|ValueToStringInternal||ValueFromStringInternal|' #     
            r'|ValueToString||ValueFromString||'
            r'ValueToFile||ValueFromFile||SaveValue|'
            r'|'
            r'|GetAP||GetDateOfAP||GetTimeOfAP' #     < >
            r'|GetDocOfAP||GetAPPosition||'
            r'SetAPToBeg||SetAPToEnd|'
            r'|DefaultChartOfAccounts||MainChartOfAccounts|' #     < >
            r'|AccountByCode||BeginOfiodBT||'
            r'EndOfiodBT||EndOfCalculatediodBT|'
            r'|SetAccount||InputChartOfAccounts||'
            r'InputSubcontoKind||MaxSubcontoCount|'
            r'|BasicCalcJournal|' #     <>
            r'|AttachAddIn||LoadAddin)\b', Keyword.Pseudo), #  
            (r'(|Export||Forward)\b', Keyword.Declaration),
            (r'~[a-zA-Z--_][a-zA-Z--0-9_]*:', Name.Label),
            (r'~[a-zA-Z--_][a-zA-Z--0-9_]*', Name.Label),
            (r'(|EndProcedure||EndFunction)\b', Keyword, '#pop'),
            (r'[a-zA-Z--_][a-zA-Z--0-9_]*', Name),
        ],
        'root': [
            include('whitespace'),
            # classes
            (r'(class|)(\s+)'                          # class keyword
             r'([a-zA-Z--_][a-zA-Z--0-9_.]*)(\s*)'   # class name
             r'(=)(\s*)'                                    # operator =
             r'([^:^{^/]*)(:{0,1})(.*?)({)',                  # class path
             bygroups(Keyword, Text, Name.Class, Text, Operator, Text, String, Operator, using(this), Keyword), 'classdef'),
            # functions
            #(r'(||procedure|function)(\s+)'
            #r'([a-zA-Z--_][a-zA-Z--0-9_]*)'
            #r'(\s*\([^;]*?\))',                           # signature
            #bygroups(Keyword, Text, Name.Function, using(this)), 'function'),
            ('', Text, 'statement'),
        ],
        'statement' : [
            include('whitespace'),
            include('statements'),
            (';', Text, '#pop'),
        ],
        'classdef': [
            include('whitespace'),
            (r'([a-zA-Z--_][a-zA-Z--0-9_.]*)(\s+)' # return arguments
             r'([a-zA-Z--_][a-zA-Z--0-9_]*)'      # method name
             r'(\s*\([^;]*?\))'                           # signature
             #r'(' + _ws + r')(;)',
             r'(;)',
             bygroups(using(this), Text, Name.Function, using(this), Text)),
            ('{', Keyword, '#push'),
            ('(}|}\s+;)', Keyword, '#pop'),
            include('statements'),
        ],
#        'function': [
#            (r'(|EndProcedure||EndFunction)\b', Keyword, '#pop'),
#            include('whitespace'),
#            include('statements'),
#            (';', Text),
#        ],
        'funcname': [
            (r'[a-z-A-Z-_][a-z-A-Z-0-9_]*', Name.Function, '#pop')
        ],
        'string': [
            (r'"', String, '#pop'),
            (r'\\([\\abfnrtv"\']|x[a-fA-F0-9]{2,4}|[0-7]{1,3})', String.Escape),
            (r'[^\\"\n]+', String), # all other characters
            (r'\\\n', String), # line continuation
            (r'\\', String), # stray backslash
        ],
        'macro': [
            (r'[^/\n]+', Comment.Preproc),
            (r'/[*](.|\n)*?[*]/', Comment),
            (r'//.*?\n', Comment, '#pop'),
            (r'/', Comment.Preproc),
            (r'(?<=\\)\n', Comment.Preproc),
            (r'\n', Comment.Preproc, '#pop'),
        ]
    }
